%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% siht_periodogram.m
%
% Implements spectral iterative hard thresholding with DFT frame
%
% Inputs:
% y - CS measurements (Mx1)
% Phi - CS matrix (MxN)
% K - Signal Sparsity
% fgran - frequency grid
% epsilon - Convergence parameter
% maxmu - Max coherence allowed
% numiter - Max number of iterations
% qfit - Quadratic fit to frequency estimates flag (binary)
%
% Outputs:
% x - output estimate
% xf - frequency estimates
% xc - frequency coefficient estimates
%
% Written by Marco F. Duarte, Program in Applied and Computational Mathematics, Princeton Univeristy
% January 2010
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [x,xf,xc] = siht_periodogram(y,Phi,K,fgran,epsilon,maxmu,numiter,qfit)

N = size(Phi,2); % Signal length
Nidx = (0:(N-1))'; % Indices
fhat = (0:fgran:N)'; % Set of testing frequencies
fhat = fhat(1:end-1);
x = zeros(N,numiter+1);
xf = []; % Signal estimate
xc = [];

res = y; % residual

iter = 1;
resnorm = 0;
while norm(res) >= norm(y)*epsilon && iter <= numiter,
    xe = x(:,iter) + Phi'*res; % Projection estimate
    % Identify estimate's off-Nyquist frequencies
    [xf,xc] = modelapprox(xe,fhat,maxmu,K,qfit);
    % Build signal estimate and residual
    for ii=1:length(xf),
        x(:,iter+1) = x(:,iter+1)+xc(ii)*exp(sqrt(-1)*2*pi*xf(ii)*Nidx/N)/sqrt(N);
    end
    res = y-Phi*x(:,iter+1);
    if resnorm == norm(res)/norm(y),
        break % Converged!
    else
        resnorm = norm(res)/norm(y);
    end
    iter = iter+1;
end
% Pick best approximation
resnorm = sum(abs(y*ones(1,size(x,2))-Phi*x).^2);
[~,idx] = min(resnorm);
x = x(:,idx);
[xf,idx] = sort(xf,'ascend');
xc = xc(idx);
