%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% modelapprox.m
%
% Perform model-based approximation for frequency sparse signals
% Assumes no two tones can have coherence larger than mu
%
% Inputs:
% xe - signal estimate input
% fhat - frequency value grid
% maxmu - maximum coherence allowed
% qfit - frequency estimate quadratic fit flag (binary)
%
% Outputs:
% tonesout - estimate component frequencies
% mu - frequency coefficients
%
% Written by Marco F. Duarte, Program in Applied and Computational Mathematics, Princeton Univeristy
% January 2010
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [tonesout,mu] = modelapprox(xe,fhat,maxmu,K,qfit)

N = floor(max(fhat))+1;
Nidx = (0:(N-1))';

% Identify possible off-Nyquist frequencies
xez = [xe; zeros(length(fhat)-N,1)];
xif = fft(xez);
xif2 = xif;
[sval,idx] = sort(abs(xif),'descend');

% Sort coefficients
toneidx = zeros(N,1);
cnt = 1;
while sval(1) > 0,
    toneidx(cnt) = idx(1); % Pick largest
    cfreq = fhat(toneidx(cnt)); % Find frequency value
    xif(toneidx(cnt)) = 0;
    xtmp = [exp(sqrt(-1)*2*pi*cfreq*Nidx/N)/sqrt(N); zeros(length(fhat)-N,1)]; % Pick frequencies within coherence factor
    ftmp = fft(xtmp)/sqrt(N);
    muidx = find(abs(ftmp)' >= maxmu);
    % Zero out corresponding off-Nyquist grid components
    xif(muidx(1):muidx(end)) = 0;
    [sval,idx] = sort(abs(xif),'descend');
    cnt = cnt+1;
end
toneidx = toneidx(toneidx ~= 0);
if length(toneidx) > K, toneidx = toneidx(1:K); end
if qfit, % Perform quadratic fit to frequency estimate
    xif = zeros(size(xif2));
    xif(toneidx) = xif2(toneidx);
    tonesout = zeros(size(toneidx));
    for cnt=1:length(toneidx),
        tmpa = fhat(max(1,(toneidx(cnt)-1)):min(length(fhat),(toneidx(cnt)+1)));
        tmpa = tmpa(:);
        tmpb = abs(xif2(max(1,(toneidx(cnt)-1)):min(length(xif),(toneidx(cnt)+1))));
        tmpb = tmpb(:);
        coefs = pinv([tmpa.^2 tmpa ones(length(tmpb),1)])*tmpb;
        if length(tmpa) == 3,
            maxf = -coefs(2)/(2*coefs(1));
            if maxf > tmpa(1) && maxf < tmpa(3), % Frequency estimate is within range
                tonesout(cnt) = maxf;
            end
        else % two or less entries in vector - pick first one.
            tonesout(cnt) = fhat(toneidx(cnt));
        end
    end
else
    tonesout = fhat(toneidx);
end
mu = zeros(size(tonesout));
for fidx = 1:length(mu), % For each frequency
    mu(fidx) = sum(xe.*exp(-sqrt(-1)*2*pi*tonesout(fidx)*(Nidx)/N))/sqrt(N); % Estimate sinusoid amplitude
end
