%% BAND_EXCLUSION generates a set of indices for exclusion due to coherence.
%
% [B] = BAND_EXCLUSION(S, COHERENCE, METHOD)
%
% BAND_EXCLUSION finds the set of indices, corresponding to some signals,
% that must be excluded to ensure incoherence between the remaining
% indices. It is based on Eqn. 12 in [1].
%
% Input variables:
%   S               The set of indices to be examined. 
%   COHERENCE       The coherence matrix for the signals corresponding to 
%                   the indices in S.
%   METHOD          Either 'single' or 'double', depending on whether
%                   single or double band-exclusion shall be used.
%
% Output variables:
%   B               The set of indices to exclude to avoid coherence with 
%                   the indices in S.
% 
% References:
%   [1]  A. Fannjiang and W. Liao, ?Coherence pattern-guided compressive
%   sensing with unresolved grids,? SIAM J. Img. Sci., vol. 5, no. 1, pp.
%   179?202, Feb. 2012.
% 
% Code implemented by: Karsten Fyhn
% Contact e-mail: kfn@es.aau.dk
%
% Copyright 2012 Karsten Fyhn
%
% Licensed under the Apache License, Version 2.0 (the "License");
% you may not use this file except in compliance with the License.
% You may obtain a copy of the License at
%
%   http://www.apache.org/licenses/LICENSE-2.0
%
% Unless required by applicable law or agreed to in writing, software
% distributed under the License is distribued on an "AS IS" BASIS,
% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
% See the License for the specific language governing permissions and
% limitations under the License.
function B = band_exclusion(S, coherence)
N = size(coherence,1);
B = zeros(N,1);
for k = S
    for j = 1:N
        mu_jk = coherence(j,k);
        if mu_jk > 0.25
            for i = 1:N
                mu_ij = coherence(i,j);
                if mu_ij > 0.25
                    B(i) = 1;
                end
            end
        end
    end
end
end