%% FREQ_ESTIMATION is the main simulation loop.
%
% [F_ERRORS, FREQ_ERRORS, COMP_TIMES] = FREQ_ESTIMATION(N, K, M, C, NUM_ITERATIONS, SNR)
%
% This main loop runs several frequency estimation frameworks for a varity
% of parameters. The output is three performance metrics: the error on the
% signal reconstruction, error on the frequency estimate and average
% computation time.
%
% Input variables:
%   N               The size of signal to use.
%   K               Sparsity of the signal.
%   M               Number of measurements to take.
%   C               Redundancy factor of the dictionary.
%   NUM_ITERATIONS  How many runs to average the results over.
%   SNR             Signal-to-noise ratio in dB.
%
% Output variables:
%   F_ERRORS        Normalized norm-2 error of signal reconstruction.
%   FREQ_ERRORS     Average cost in frequency estimation using Hungarian
%                   algorithm.
%   COMP_TIMES      Average computation times.
% 
% Code implemented by: Karsten Fyhn and Hamid Dadkhahi
% Contact e-mail: kfn@es.aau.dk
%
% Copyright 2012 Karsten Fyhn and Hamid Dadkhahi
%
% Licensed under the Apache License, Version 2.0 (the "License");
% you may not use this file except in compliance with the License.
% You may obtain a copy of the License at
%
%   http://www.apache.org/licenses/LICENSE-2.0
%
% Unless required by applicable law or agreed to in writing, software
% distributed under the License is distribued on an "AS IS" BASIS,
% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
% See the License for the specific language governing permissions and
% limitations under the License.
function [f_errors, freq_errors, comp_times] = freq_estimation(n, k, M, C, num_iterations,SNR)
methods = 7;

% window size for MUSIC
win_size = 5;

% normalized norm-2 errors in recovery of time-domain signals
f_errors = zeros(methods, length(C),length(M),num_iterations);

% errors in frequency estimation as a cost function of sum of frequency differences
% between true frequencies and estimated frequencies
freq_errors = zeros(methods, length(C),length(M),num_iterations);

comp_times = zeros(methods, length(C),length(M),num_iterations);

fprintf(1,'##############\n');
fprintf(1, 'Progress: %3d%%', 0);
for it = 1 : num_iterations   
    for i = 1 : length(C)
        c = C(i);
        p = c * n;
        
        % redundant dictionary
        D = zeros(n,p);
        for a = 1 : n
            for b = 1 : p
                D(a,b) = exp(1i*2*pi*a*b/p) / sqrt(n);
            end
        end
        coherence = abs(D'*D);
               
        % Generate signal
        tone_sep = 1; % This is the width of the main lobe of the Dirilecht kernel
        true_freqs = sort(gen_ks(k, n, tone_sep));
        f = signal(true_freqs, n);
                        
        for m = 1 : length(M)        
            % measurement matrix
            A = randn(M(m),n);
            Q = zeros(M(m),n);
            R = zeros(M(m),M(m));
            for kk = 1:M(m)
                v = A(kk,:);
                for ll = 1:kk-1
                    R(ll,kk) = Q(ll,:)*A(kk,:)';
                    v = v - R(ll,kk)*Q(ll,:);
                end
                R(kk,kk) = norm(v);
                Q(kk,:) = v/R(kk,kk);
            end
            A = Q;
            y = A * f;
            
            if isempty(SNR)
                y_noisy = y;
                epsilon = 10^(-10);
            else               
                y_power = mean(abs(y).^2);
                noise_power = y_power / 10^(SNR/10);
                noise = sqrt(noise_power/2) * (randn(size(y)) + 1i*randn(size(y)));
                y_noisy = y + noise;
                epsilon = norm(noise,2)/norm(y,2);
                
            end            
                        
            % analysis
            tic;
            if isempty(SNR)
                f_hat = analysis_eq(y,A,D);
            else
                f_hat = analysis_cs(y_noisy,A,D,epsilon);
            end
            [~,freq_analysis,~] = musicapprox(f_hat,k,win_size);
            comp_times(1,i,m,it) = toc;
            f_errors(1,i,m,it) = norm(f_hat - f,2)/norm(f,2);
            freq_analysis_matrix = gen_matrix_hungarian(true_freqs,freq_analysis);
            [~,freq_errors(1,i,m,it)] = Hungarian(freq_analysis_matrix);
                        
            % synthesis
            tic;
            if isempty(SNR)
                x_hat = synthesis_eq(y,A,D);                
            else
                x_hat = synthesis_cs(y_noisy,A,D,epsilon);
            end
            [~,freq_synthesis,~] = musicapprox(D*x_hat,k,win_size);
            comp_times(2,i,m,it) = toc;
            
            f_errors(2,i,m,it) = norm(D*x_hat - f,2)/norm(f,2);
            freq_synthesis_matrix = gen_matrix_hungarian(true_freqs,freq_synthesis);
            [~,freq_errors(2,i,m,it)] = Hungarian(freq_synthesis_matrix);
                       
            % SIHT
            numiter = 100;
            maxmu = 0.1;
            qfit = 1;
            tic;
            [f_siht,~,~] = siht_periodogram(y_noisy,A,k,n/p,epsilon,maxmu,numiter,qfit);
            
            [~,xf,~] = musicapprox(f_siht,k,win_size);
            
            comp_times(3,i,m,it) = toc;
            
            f_errors(3,i,m,it) = norm(f_siht - f,2)/norm(f,2);
            freq_siht = xf;
            freq_siht_matrix = gen_matrix_hungarian(true_freqs,freq_siht);
            [~,freq_errors(3,i,m,it)] = Hungarian(freq_siht_matrix);
            
            % SDP
%             indices = randperm(n);
%             indices = sort(indices(1:M(m)));
%             A_cs = eye(n);
%             A_cs = A_cs(indices,:);
% 
%             if isempty(SNR)
%                 xVals = A_cs * f;
%             else
%                 xVals = A_cs * f + noise;                
%             end
%             tic;
%             [x_sdp,~,~] = ctscs_sdpt3(xVals,indices,n);
%             [~,freq_sdp,~] = musicapprox(x_sdp,k,win_size);
%             comp_times(4,i,m,it) = toc;
%             
%             f_errors(4,i,m,it) = norm(x_sdp - f,2)/norm(f,2);
%             freq_sdp_matrix = gen_matrix_hungarian(true_freqs,freq_sdp);
%             [~,freq_errors(4,i,m,it)] = Hungarian(freq_sdp_matrix);
                                    
            % BOMP
            tic;
            [sig2, freq_bomp] = bomp(y_noisy, A, D, k, 1/c, coherence);
            comp_times(5,i,m,it) = toc;
            
            f_errors(5,i,m,it) = norm(sig2 - f,2)/norm(f,2);
            freq_omp_matrix = gen_matrix_hungarian(true_freqs,freq_bomp);
            [~,freq_errors(5,i,m,it)] = Hungarian(freq_omp_matrix);
            
            % CBP
            bs = 1/c:1/c:n;
            if isempty(SNR)
                NOISY = 0;
            else
                NOISY = 1;
            end
            tic;
            [sig1, freq_pocvx] = cbp(y_noisy, A, bs, k, 1/c, epsilon, NOISY);
            comp_times(6,i,m,it) = toc;
            
            f_errors(6,i,m,it) = norm(sig1 - f,2)/norm(f,2);
            freq_pocvx_matrix = gen_matrix_hungarian(true_freqs,freq_pocvx);
            [~,freq_errors(6,i,m,it)] = Hungarian(freq_pocvx_matrix);
            
            % BISP
            tic;
            [sig2, freq_bisp] = bisp(y_noisy, A, D, k, coherence, epsilon, NOISY);
            comp_times(7,i,m,it) = toc;
            
            f_errors(7,i,m,it) = norm(sig2 - f,2)/norm(f,2);
            freq_bisp_matrix = gen_matrix_hungarian(true_freqs,freq_bisp);
            [~,freq_errors(7,i,m,it)] = Hungarian(freq_bisp_matrix);
            
            % Status
            fprintf(1, '\b\b\b\b%3d%%', round(100*(m+length(M)*(i-1)+length(C)*length(M)*(it-1))/(num_iterations*length(C)*length(M))));
        end
    end
end
fprintf(1, '\n');

% averaging errors in signal recovery over all experiments
%f_errors = sum(f_errors,4) / num_iterations;

% averaging errors in frequency estimation over all experiments
%freq_errors = sum(freq_errors,4) / num_iterations;
end