%% CBP implements the CBP algorithm.
%
% [SIG, FREQS_EST] = CBP(F, A, BS, K, DELTA, EPSILON, NOISY)
%
% This the implementation of the CBP algorithm from [1], modified for
% frequency estimation.
%
% Input variables:
%   F               The measurements.
%   A               The measurement matrix
%   BS              The frequencies to generate a dictionary from.
%   K               The sparsity of the signal F.
%   DELTA           The spacing between the frequencies.
%   EPSILON         The noise uncertainty.
%   NOISY           Binary variable signifying whether the signal is noisy
%                   or not.
%
% Output variables:
%   SIG             The reconstructed signal.
%   FREQS_EST       The estimated frequencies in the signal.
% 
% References:
%   [1] C. Ekanadham, D. Tranchina, and E. P. Simoncelli, ?Recovery of
%   sparse translation-invariant signals with continuous basis pursuit?,
%   IEEE Transactions on Signal Processing, vol. 59, no. 10, pp. 4735?4744,
%   Oct. 2011.

% Code implemented by: Karsten Fyhn
% Contact e-mail: kfn@es.aau.dk
%
% Copyright 2012 Karsten Fyhn
%
% Licensed under the Apache License, Version 2.0 (the "License");
% you may not use this file except in compliance with the License.
% You may obtain a copy of the License at
%
%   http://www.apache.org/licenses/LICENSE-2.0
%
% Unless required by applicable law or agreed to in writing, software
% distributed under the License is distribued on an "AS IS" BASIS,
% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
% See the License for the specific language governing permissions and
% limitations under the License.
function [sig, freqs_est] = cbp(f, A, bs, k, delta, epsilon, NOISY)
n = size(A,2);
d = length(bs);
ns = 1:n;

% Make polar interpolation dictionaries, C, U and V
poDicC = zeros(n,d);
poDicU = zeros(n,d);
poDicV = zeros(n,d);

for ii = 1:d
    Funcs = zeros(3,n);
    Funcs(1,:) = exp(1j*2*pi*(bs(ii)-delta/2)*ns/n)/sqrt(n);
    Funcs(2,:) = exp(1j*2*pi*bs(ii)*ns/n)/sqrt(n);
    Funcs(3,:) = exp(1j*2*pi*(bs(ii)+delta/2)*ns/n)/sqrt(n);
    r = norm(Funcs(1,:),2);
    theta = acos(real(Funcs(1,:)*Funcs(2,:)'));
    rotMtx = [  1., r*cos(theta),    -r*sin(theta);
        1., r,                  0.;
        1., r*cos(theta),    r*sin(theta)];
    rotMtx = inv(rotMtx);
    poDicC(:,ii) = rotMtx(1,:)*Funcs;
    poDicU(:,ii) = rotMtx(2,:)*Funcs;
    poDicV(:,ii) = rotMtx(3,:)*Funcs;
end

AC = A*poDicC;
AU = A*poDicU;
AV = A*poDicV;

AAC = [AC AU AV];
b = [ones(1,d) zeros(1,2*d)];
Ai = {}; bi = {}; ci = {}; di = {}; ei = {}; fi = {};
for ii=1:d,
    tmp = zeros(3,3*d); tmp(1,ii) = 1; tmp(2,ii+d) = 1; tmp(3,ii+2*d) = 1;
    Ai{ii} = tmp;
    tmp = zeros(1,3*d); tmp(ii) = -r*cos(theta); tmp(ii+d) = 1;
    bi{ii} = tmp;
    tmp = zeros(1,3*d); tmp(ii) = 1;
    ci{ii} = tmp;
    tmp = zeros(1,3*d); tmp(ii) = r; tmp(ii+d) = -1;
    di{ii} = tmp;
end
Aa = zeros(1,3*d);

cvx_quiet(1);
cvx_precision best;
if NOISY
    cvx_begin
        variable xx(3*d);
        % xx is the concatenation of x, y, and z
        % variable x(d);
        % variable y(d);
        % variable z(d);
        % Rewrote the objective function by concatenating
        % minimize( 1/(2*variance)*norm(y_noisy-AC*x-AU*y-AV*z,2) + lambda*norm(x,1) );

        %minimize( 1/(2*epsilon*norm(f,2))* norm(f-AAC*xx,2) + b*xx );
        minimize( b*xx );
        subject to
            norm(f-AAC*xx,2) <= epsilon*norm(f,2);
            for ii = 1:d
                % Each one of the constraints is now written in the form
                % norm(matrix*xx,2) <= vector*xx
                % matrices and vectors are defined before the cvx code
                % If I need a 0 on either side of the inequality, I use a zero
                % matrix/vector Aa.
                % x(ii) >= 0;
                norm(Aa*xx,2) <= ci{ii}*xx;
                % y(ii)^2 + z(ii)^2 <= x(ii)^2*r^2;
                norm(Ai{ii}*xx,2) <= (ci{ii}*sqrt(r^2+1))*xx;
                % x(ii)*r*cos(theta) <= y(ii) <= x(ii)*r;
                norm(Aa*xx,2) <= bi{ii}*xx;
                norm(Aa*xx,2) <= di{ii}*xx;
            end
    cvx_end
else
    cvx_begin
        variable xx(3*d);
        % xx is the concatenation of x, y, and z
        % variable x(d);
        % variable y(d);
        % variable z(d);
        % Rewrote the objective function by concatenating
        % minimize( 1/(2*variance)*norm(y_noisy-AC*x-AU*y-AV*z,2) + lambda*norm(x,1) );

        minimize( 1/(2*epsilon*norm(f,2))* norm(f-AAC*xx,2) + b*xx );
        %minimize( b*xx );
        subject to
            %norm(f-AAC*xx,2) <= epsilon*norm(f,2);
            for ii = 1:d
                % Each one of the constraints is now written in the form
                % norm(matrix*xx,2) <= vector*xx
                % matrices and vectors are defined before the cvx code
                % If I need a 0 on either side of the inequality, I use a zero
                % matrix/vector Aa.
                % x(ii) >= 0;
                norm(Aa*xx,2) <= ci{ii}*xx;
                % y(ii)^2 + z(ii)^2 <= x(ii)^2*r^2;
                norm(Ai{ii}*xx,2) <= (ci{ii}*sqrt(r^2+1))*xx;
                % x(ii)*r*cos(theta) <= y(ii) <= x(ii)*r;
                norm(Aa*xx,2) <= bi{ii}*xx;
                norm(Aa*xx,2) <= di{ii}*xx;
            end
    cvx_end
end
x = xx(1:d); y = xx(d+(1:d)); z = xx(2*d+(1:d));
y = y.*x*r./sqrt(y.^2 + z.^2);
z = z.*x*r./sqrt(y.^2 + z.^2);

sig = poDicC*x + poDicU*y + poDicV*z;

[~, idxs] = sort(x,1,'descend');
freqs_est = zeros(k,1);
if sum(x) > 10^(-10)
    for ii = 1:k
        y_ii = y(idxs(ii));
        z_ii = z(idxs(ii));
        freqs_est(ii) = bs(idxs(ii)) + atan(z_ii/y_ii)*delta/(2*theta);
        if isnan(freqs_est(ii))
            freqs_est(ii) = 0;
        end
    end
end
end