%% BISP implements the BISP algorithm.
%
% [SIG, FREQS_EST] = BISP(Y, PHI, PSI, K, COHERENCE, EPSILON, NOISY)
%
% This the implementation of the Band-excluded Interpolation Subspace
% Pursuit (BISP) algorithm.
%
% Input variables:
%   Y               The measurements.
%   PHI             The measurement matrix
%   PSI             The dictionary.
%   K               The sparsity of the signal F.
%   COHERENCE       The coherence matrix used for band-exclusion.
%   EPSILON         The noise uncertainty.
%   NOISY           Binary variable signifying whether the signal is noisy
%                   or not.
%
% Output variables:
%   SIG             The reconstructed signal.
%   FREQS_EST       The estimated frequencies in the signal.
% 
% Code implemented by: Karsten Fyhn
% Contact e-mail: kfn@es.aau.dk
%
% Copyright 2012 Karsten Fyhn
%
% Licensed under the Apache License, Version 2.0 (the "License");
% you may not use this file except in compliance with the License.
% You may obtain a copy of the License at
%
%   http://www.apache.org/licenses/LICENSE-2.0
%
% Unless required by applicable law or agreed to in writing, software
% distributed under the License is distribued on an "AS IS" BASIS,
% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
% See the License for the specific language governing permissions and
% limitations under the License.
function [sig, freqs_est] = bisp(y, Phi, Psi, K, coherence, epsilon, NOISY)
    N = size(Psi,1);
    P = size(Psi,2);
    A = Phi*Psi;
    freqs_est = zeros(K,1);
    delta = N / P;
    
    % Initialization
    proxy = abs(A'*y);
    pruned_proxy = proxy;
    atoms = [];
    for ii=1:K
        BE = band_exclusion(atoms, coherence);
        pruned_proxy(BE==1) = 0;
        sorted_proxy = sort(abs(pruned_proxy),1,'descend');
        atoms = union(atoms, find(abs(pruned_proxy)==sorted_proxy(1)));
    end
    S = atoms;
    [~, sorted_idxs] = sort(abs(proxy(S)),1,'descend');
    S = S(sorted_idxs);
    yr = y - A(:,S) * (A(:,S) \ y); 
        
    % Iterations
    for ii = 1:K
        proxy = abs(A'*yr);
        pruned_proxy = proxy;
        for jj=1:K
            BE = band_exclusion(S, coherence);
            pruned_proxy(BE==1) = 0;
            sorted_proxy = sort(abs(pruned_proxy),1,'descend');
            S = union(S, find(abs(pruned_proxy)==sorted_proxy(1)));
        end
        alpha = zeros(P,1);
        alpha(S) = A(:,S) \ y;
        sorted_alpha = sort(abs(alpha),1,'descend');
        threshold = sorted_alpha(K+1);
        S = find(abs(alpha)>threshold);
        [~, sorted_idxs] = sort(abs(alpha(S)),1,'descend');
        S = S(sorted_idxs);
        
        % Polar interpolation
        B = [];
        for jj=1:K
            B = union(B, (S(jj)-1):(S(jj)+1));
        end
        [sig, freqs_est] = cbp(y, Phi, B*delta, K, delta, epsilon, NOISY);
        if sum(freqs_est==0)
            NOISY = ~NOISY;
            [sig, freqs_est] = cbp(y, Phi, B*delta, K, delta, epsilon, NOISY);
        end
        for jj=1:K
            if freqs_est(jj) == 0
                freqs_est(jj) = S(jj)*delta;
            end
            sig = signal(freqs_est, N);
        end
                
        yr_new = y - Phi*sig;
        if norm(yr_new,2) > norm(yr,2)
            break
        else
            yr = yr_new;
        end
    end
end
