%% BOMP is the implementation of the BOMP algorithm.
%
% [SIG, TAUS_EST] = BOMP(T, Y, PHI, PSI, K, DELTA, COHERENCE)
%
% The implementation of the Band-Excluded Orthogonal Matching Pursuit (IBOMP)
% algorithm from [1].
%
% Input variables:
%   T               The vector of time samples.
%   Y               The received signal.
%   PHI             The CS measurement matrix.
%   PSI             The dictionary matrix.
%   K               The number of delays to estimate.
%   COHERENCE       The coherence matrix between atoms in the dictionary.
%
% Output variables:
%   SIG             The reconstructed signal.
%   TAUS_EST        The estimated delays.
%
% References:
%   [1]  A. Fannjiang and W. Liao, ?Coherence pattern-guided compressive
%   sensing with unresolved grids,? SIAM J. Img. Sci., vol. 5, no. 1, pp.
%   179-202, Feb. 2012.
%
% Code implemented by: Karsten Fyhn
% Contact e-mail: kfn@es.aau.dk
%
% Copyright 2012 Karsten Fyhn
%
% Licensed under the Apache License, Version 2.0 (the "License");
% you may not use this file except in compliance with the License.
% You may obtain a copy of the License at
%
%   http://www.apache.org/licenses/LICENSE-2.0
%
% Unless required by applicable law or agreed to in writing, software
% distributed under the License is distribued on an "AS IS" BASIS,
% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
% See the License for the specific language governing permissions and
% limitations under the License.
function [sig, freqs_est] = bomp(y, Phi, Psi, K, delta, coherence)
    N = size(Phi,2);
    yr = y;
    A = Phi*Psi;
    B = zeros(N, K);
    alpha = zeros(K,1);
    freqs_est = zeros(K,1);
    S = [];
    for ii = 1:K
        proxy = abs(A'*yr);
        BE = band_exclusion(S, coherence);
        proxy(BE==1) = 0;
        [~, k_ii] = max(proxy);
        freqs_est(ii) = k_ii*delta;
        B(:,ii) = signal(freqs_est(ii), N);
        alpha(1:ii) = (Phi*B(:,1:ii)) \ y;
        comp_sig = Phi*B*alpha;
        yr = y-comp_sig;
        S = cat(2, S, k_ii);
    end
    sig = B*alpha;
    return
end