/*
    PacketBench - Workload Characterization Tool
    Copyright (C) 2004  Ramaswamy Ramaswamy

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "bench.h"
#include "ipv4.h"

/*
 * Check an IP header checksum.
 * "To check a checksum, the 1's complement sum is computed over the
 * same set of octets, including the checksum field.  If the result
 * is all 1 bits (-0 in 1's complement arithmetic), the check
 * succeeds." (RFC 1071, Computing the Internet Checksum)
 * 
 * Obtained from print-ip.c, tcpdump-3.7.1
 *
 */
static inline unsigned short in_cksum(const unsigned short *addr, unsigned int len, int csum)
{
  int nleft = len;
  const unsigned short *w = addr;
  unsigned short answer;
  int sum = csum;
  
  /*
   *  Our algorithm is simple, using a 32 bit accumulator (sum),
   *  we add sequential 16 bit words to it, and at the end, fold
   *  back all the carry bits from the top 16 bits into the lower
   *  16 bits.
   */

  while (nleft > 1)  
    {
      sum += *w++;            
      nleft -= 2;
    }
  if (nleft == 1)
    sum += htons(*(u_char *)w<<8);
  
  /*
   * add back carry outs from top 16 bits to low 16 bits
   */

  sum = (sum >> 16) + (sum & 0xffff);     /* add hi 16 to low 16 */
  sum += (sum >> 16);                     /* add carry */
  answer = ~sum;                          /* truncate to 16 bits */
  return (answer);
}

/*
 * Validates packet according to RFC 1812
 * Returns 1 --> packet is valid
 * Returns 0 --> Invalid packet
 */

static int validate_packet(struct ip *iphdr,int ll_length)
{
  /*
   * Check link layer length > minimum ip datagram length
   * Validate packet checksum (RFC 1071)
   * Check IP version = 4
   * Check if header length > minimum header length (5 words)
   * Check if packet length > minimum header length
   */
                                                                                                         
  /*
   * This causes changes in instruction count ~ 60 instr. ?
   */
  /*
  in_cksum((const unsigned short *)iphdr,(iphdr->ip_hl*4),0);
                                                                                                         
  if (
       (ll_length > MIN_IP_DATAGRAM) &&
       (iphdr->ip_v == 4) &&
       (iphdr->ip_hl >= MIN_IP_DATAGRAM/4) &&
       (ntohs(iphdr->ip_len) >= iphdr->ip_hl)
     )
    return 1;
  else
    return 0;
  */
                                                                                                         
   if (
       (ll_length > MIN_IP_DATAGRAM) &&
       (in_cksum((const unsigned short *)iphdr,(iphdr->ip_hl*4),0)) &&
       //Checksum fails with TSH files.
       //Use the line below for TCPDUMP files and the line above for TSH
       //files
       //!(in_cksum((const unsigned short *)iphdr,(iphdr->ip_hl*4),0)) &&
       (iphdr->ip_v == 4) &&
       (iphdr->ip_hl >= MIN_IP_DATAGRAM/4) &&
       (ntohs(iphdr->ip_len) >= iphdr->ip_hl)
     )
    return 1;
  else
    return 0;
}

/*******************************************************************
 * Function that performs IPv4 routing                             *
 * Return value = 0 --> packet has been processed,queue it locally *
 * Return value = 1 --> packet has been processed,dump to file     *
 * Assumptions :                                                   *
 * 1) No IP options processing                                     *
 * 2) Packets are not fragmented/re-assembled                      *
 * 3) Packet is always forwarded                                   *
 * 4) No ICMP redirects                                            *
 *******************************************************************/

int ipv4routing(packet *pass_packet)
{  
  struct ip *iphdr;
  char *daddr;

  /*
   * Have we got an IP packet ?
   * If so, do routing, otherwise drop to trace file
   */

  if (pass_packet->meta_buffer.ll_type != 0x800)
    {
#ifdef VERBOSE
      printf("\nNon-IP packet received, packet dropped");
      printf("\nType : 0x%x",pass_packet->meta_buffer.ll_type);
      printf("\nLength : %d",pass_packet->meta_buffer.ll_length);
#endif
      switch(trace_format)
        {
        case 0 : 
          write_packet_to_tcpdump_file(pass_packet,DROP);
          break;
        case 1 :
          write_packet_to_tsh_file();
          break;
        }      
      return 1;
    }

  /*
   * Extract the IP header from the packet
   */
  iphdr = (struct ip*)(pass_packet->data);

  /* 
   * Validate packet contents according to RFC 1812
   */
  if (validate_packet(iphdr,pass_packet->meta_buffer.ll_length))
    {
      /* 
       * Check and decrement TTL by 1
       */
      if (iphdr->ip_ttl == 0)
	{
#ifdef VERBOSE
	  printf("\n%s:%d(%s), Packet TTL is zero! Discarding...",__FILE__,__LINE__,__FUNCTION__);
#endif
	  switch(trace_format)
	    {
	    case 0 : 
	      write_packet_to_tcpdump_file(pass_packet,DROP);
	      break;
	    case 1 :
	      write_packet_to_tsh_file();
	      break;
	    }      
	}
      else
	{
	  /*	   
	   * Decrement TTL by IPDECTTL units. Incrementally
	   * recompute the checksum. Code obtained from
	   * NetBSD source, netinet/ip_flow.c
	   */
	  iphdr->ip_ttl -= IPDECTTL;
	  if (iphdr->ip_sum >= (unsigned short) ~htons(IPDECTTL << 8))
	    iphdr->ip_sum -= ~htons(IPDECTTL << 8);
	  else
	    iphdr->ip_sum += htons(IPDECTTL << 8);	  
	  
	  /* 
	   * Print some stuff from the packet
	   * to maintain sanity
	   */
	  daddr = inet_ntoa(iphdr->ip_dst);
#ifdef VERBOSE
	  printf("\nPacket valid : ");
	  printf("\nType : 0x%x",pass_packet->meta_buffer.ll_type);
	  printf("\nLength : %d",pass_packet->meta_buffer.ll_length);
	  printf("\nUpdated Checksum : 0x%x",ntohs(iphdr->ip_sum));
	  printf("\nTTL : %d", iphdr->ip_ttl);
	  printf("\nDestination Address : %s",daddr);
#endif     

	  /*
	   * Lookup the destination address in the radix tree
	   * Get the port number to which it has to be forwarded
	   * 0.0.0.0 -> default route
	   */

	  lookup_tree(daddr);

	  /*
	   * Write packet to dump file
	   */
	  switch(trace_format)
	    {
	    case 0 : 
	      write_packet_to_tcpdump_file(pass_packet,DUMP);
	      break;
	    case 1 :
	      write_packet_to_tsh_file();
	      break;
	    }      
	}
    }
  else
    {
#ifdef VERBOSE
      printf("\n%s:%d(%s), Invalid packet! Discarding...",__FILE__,__LINE__,__FUNCTION__);
#endif
      switch(trace_format)
        {
        case 0 : 
          write_packet_to_tcpdump_file(pass_packet,DROP);
          break;
        case 1 :
          write_packet_to_tsh_file();
          break;
        }      
    }

  /*
   * Function will return 0 if it needs to store the packet in queue
   * Queue pointers are modified based on return value -> ? FIXME ?
   */
}

