/*
    PacketBench - Workload Characterization Tool
    Copyright (C) 2004  Ramaswamy Ramaswamy

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "bench.h"
#include "ipsec.h"

/*
 * Check an IP header checksum.
 * "To check a checksum, the 1's complement sum is computed over the
 * same set of octets, including the checksum field.  If the result
 * is all 1 bits (-0 in 1's complement arithmetic), the check
 * succeeds." (RFC 1071, Computing the Internet Checksum)
 * 
 * Obtained from print-ip.c, tcpdump-3.7.1
 *
 */
static inline unsigned short in_cksum(const unsigned short *addr, unsigned int len, int csum){
  int nleft = len;
  const unsigned short *w = addr;
  unsigned short answer;
  int sum = csum;
  
  /*
   *  Our algorithm is simple, using a 32 bit accumulator (sum),
   *  we add sequential 16 bit words to it, and at the end, fold
   *  back all the carry bits from the top 16 bits into the lower
   *  16 bits.
   */

  while (nleft > 1)  
    {
      sum += *w++;            
      nleft -= 2;
    }
  if (nleft == 1)
    sum += htons(*(u_char *)w<<8);
  
  /*
   * add back carry outs from top 16 bits to low 16 bits
   */

  sum = (sum >> 16) + (sum & 0xffff);     /* add hi 16 to low 16 */
  sum += (sum >> 16);                     /* add carry */
  answer = ~sum;                          /* truncate to 16 bits */
  return (answer);
}

/*******************************************************************
 * Function that performs ipsec encryption                         *
 * Return value = 0 --> packet has been processed,queue it locally *
 * Return value = 1 --> packet has been processed,dump to file     *
 * Assumptions :                                                   *
 * We do ESP tunneling mode IPSEC with 3DES                         *
 * Outbound packet processing (encryption) is performed            *
 *******************************************************************/

int ipsec(packet *pass_packet)
{  
  static unsigned int seq_no;
  struct ip *iphdr;
  char *daddr;
  unsigned short padding;
  int i=0;
  int bufsize;

  /* Outer encapsulating IP header */
  struct ip outer_iphdr;

  /* ESP header - comes before encrypted data */
  esp_header esphdr;

  /*
   * Extract the "inner" IP header from the packet
   */
  iphdr = (struct ip*)(pass_packet->data);

  /*	   
   * Decrement TTL by IPDECTTL units. Incrementally
   * recompute the checksum. Code obtained from
   * NetBSD source, netinet/ip_flow.c
   */
  iphdr->ip_ttl -= IPDECTTL;
  if (iphdr->ip_sum >= (unsigned short) ~htons(IPDECTTL << 8))
    iphdr->ip_sum -= ~htons(IPDECTTL << 8);
  else
    iphdr->ip_sum += htons(IPDECTTL << 8);	  
  

  /*
   * Construct the outer IP header
   * Most of outer header is copied from
   * inner header. This is not RFC compliant
   * needs to be fixed
   * See RFC 2401 for details
   */

  outer_iphdr.ip_v = 4;
  outer_iphdr.ip_hl = iphdr->ip_hl; // FIXME
  outer_iphdr.ip_tos = iphdr->ip_tos;
  outer_iphdr.ip_len = iphdr->ip_len; // FIXME
  outer_iphdr.ip_id = iphdr->ip_id; // FIXME
  outer_iphdr.ip_off = iphdr->ip_off; // FIXME
  outer_iphdr.ip_ttl = 255; // FIXME
  outer_iphdr.ip_p = ESP;
  outer_iphdr.ip_sum = iphdr->ip_sum; // FIXME
  outer_iphdr.ip_src.s_addr = htonl(0xdeaddead); // dummy
  outer_iphdr.ip_dst.s_addr = htonl(0xdeaddead); // dummy

  /*
   * Construct the ESP header
   * See RFC 2406 for details
   */

  esphdr.spi = SPIINDEX; // fixed for a given trace in ipsec.h
  esphdr.seq_no = seq_no; // incremented per packet
  memcpy(esphdr.iv,enccipher.iv,MAX_IV_SIZE);   // Copy IV from cipher.

  /* 
   * Print some stuff from the packet
   * to maintain sanity
   */

#ifdef VERBOSE
  printf("\nPacket : ");
  printf("\nType : 0x%x",pass_packet->meta_buffer.ll_type);
  printf("\nLength : %d bytes",pass_packet->meta_buffer.ll_length);
  printf("\nInner destination Address : %s",inet_ntoa(iphdr->ip_dst));
  printf("\nSPI : %d",esphdr.spi);
  printf("\nSequence no : %d",esphdr.seq_no);
  printf("\nNew (outer) TTL : %d",outer_iphdr.ip_ttl);
  printf("\nNew (outer) source address: %s",inet_ntoa(outer_iphdr.ip_src));
  printf("\nNew (outer) destination address: %s",inet_ntoa(outer_iphdr.ip_dst));
#endif     
  
  /* Packet length is calculated as link layer length minus the size
   * of the ethernet header */
  bufsize = (pass_packet->meta_buffer.ll_length - ETHER_SIZE);

  /* Compute length of padding bytes required */
  padding = DES_BLOCK_SIZE - (bufsize%DES_BLOCK_SIZE);

#ifdef VERBOSE
  printf("\nBuffer length = %d bytes, padding = %d bytes",bufsize,padding);
#endif

  /* Pad the block so that it is aligned on a 16 byte boundary */
  /* Last 2 bytes are padding length and inner header protocol type */
  for (i=0;i<padding-2;i++)
    {
      *(pass_packet->data+i+bufsize)=0;
    }

  /* Pad last 2 bytes with padding length and inner header protocol type*/
  /* ip within ip = 04 */
  /* This will NOT work if padding = 1. I am running out of time. FIXME */
  if (padding == 1)
    {
      *(pass_packet->data+i+bufsize)=4;
    }
  else
    {
      *(pass_packet->data+i+bufsize)=padding-1;
      *(pass_packet->data+i+1+bufsize)=4;
    }

#ifdef VERBOSE
  printf("\n\n");

  for(i=0;i<bufsize+padding;i++)
    {
      printf("%02x",pass_packet->data[i]);
      if ((i%4)==3) printf(" ");
      if ((i%16)==15) printf("\n");
    }
  printf("\n");
#endif

  /* 
   * Encrypt payload. CBC mode. IV used from cipher
   * Input and output buffers are the same
   */

#ifdef VERBOSE
  printf("\nEncrypting");
#endif

  if (DES3_CBCUpdate(&enccipher,pass_packet->data,pass_packet->data,(bufsize+padding)) == 0)
    {
#ifdef VERBOSE
      printf("...%d bytes encoded",bufsize+padding);
#endif
    }
  else
    {
#ifdef VERBOSE
      printf("...Failure");
#endif
    }

  /* 
   * Shove outer IP header and ESP header in front of packet contents
   * Ouch! FIXME
   */
  memmove(pass_packet->data+IP_SIZE+ESP_SIZE,pass_packet->data,bufsize+padding);
  
  memcpy(pass_packet->data,&outer_iphdr,IP_SIZE);
  memcpy(pass_packet->data+IP_SIZE,&esphdr,ESP_SIZE);

  /* Update link layer length */
  pass_packet->meta_buffer.ll_length = bufsize+padding+IP_SIZE+ESP_SIZE+ETHER_SIZE;

#ifdef VERBOSE
  printf("\n");

  for(i=0;i<bufsize+padding+IP_SIZE+ESP_SIZE;i++)
    {
      printf("%02x",pass_packet->data[i]);
      if ((i%4)==3) printf(" ");
      if ((i%16)==15) printf("\n");
    }
  printf("\n\n");
#endif

  /* Increment sequence number for ESP header */
  seq_no++;

  /*
   * Write packet to dump file
   */
  switch(trace_format)
    {
    case 0 : 
      write_packet_to_tcpdump_file(pass_packet,DUMP);
      break;
    case 1 :
      write_packet_to_tsh_file();
      break;
    }     
  return 1;
  
  /*
   * Function will return 0 if it needs to store the packet in queue
   * Queue pointers are modified based on return value -> ? FIXME ?
   */
}

