/*
    PacketBench - Workload Characterization Tool
    Copyright (C) 2004  Ramaswamy Ramaswamy

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "bench.h"
#include "flowid.h"

/* IP Flow Classification */

// Instrumentation variables
static int numsyns; // Number of SYNS
static int numfins; // Number of FINS
static int numrsts; // Number of RESETS
static int tcp_count; // Number of TCP packets
static int udp_count; // Number of UDP packets
static int other_count; // Number of "other" packets
static int collisons; // Number of collisons

void init_flowid()
{
  // Allocate and initialize memory for 3 lists
  // 1. Hash table
  // 2. Free list for flow records
  // 3. Free list for five tuple records

  // Size of all three data structures are the same = IPFLOW_HASHSIZE

  int i;
  struct flow_record *flow_temp;
  struct fivetuple *tuple_temp;

#ifdef GLOBAL_PACKET_MEMORY
  hash_table = temp_hash_table;
  free_flow_list = temp_free_flow_list;
  free_fivetuple_list = temp_free_fivetuple_list;
#endif

#ifndef GLOBAL_PACKET_MEMORY
  hash_table = (struct hash_entry *)malloc(IPFLOW_HASHSIZE * sizeof(struct hash_entry));
  
  if (hash_table == NULL)
    {
      printf("\n%s:%d(%s),Unable to allocate memory for hash table",__FILE__,__LINE__,__FUNCTION__);
      exit(0);
    }
#endif

#ifdef VERBOSE
  printf("\n\n%s:%d(%s),Allocated memory for hash table",__FILE__,__LINE__,__FUNCTION__);
#endif

#ifndef GLOBAL_PACKET_MEMORY
  free_flow_list = (struct flow_record *)malloc(IPFLOW_HASHSIZE * sizeof(struct flow_record));
  
  if (free_flow_list == NULL)
    {
      printf("\n%s:%d(%s),Unable to allocate memory for flow list",__FILE__,__LINE__,__FUNCTION__);
      exit(0);
    }
#endif

#ifdef VERBOSE
  printf("\n\n%s:%d(%s),Allocated memory for flow list",__FILE__,__LINE__,__FUNCTION__);
#endif

#ifndef GLOBAL_PACKET_MEMORY
  free_fivetuple_list = (struct fivetuple *)malloc(IPFLOW_HASHSIZE * sizeof(struct fivetuple));
  
  if (free_fivetuple_list == NULL)
    {
      printf("\n%s:%d(%s),Unable to allocate memory for five tuple list",__FILE__,__LINE__,__FUNCTION__);
      exit(0);
    }
#endif
  
#ifdef VERBOSE
  printf("\n\n%s:%d(%s),Allocated memory for five tuple list",__FILE__,__LINE__,__FUNCTION__);
#endif

  // Link the next pointers in the free lists

  flow_temp = free_flow_list;
  tuple_temp = free_fivetuple_list;

  for(i=0;i<IPFLOW_HASHSIZE-1;i++)
    {
      (flow_temp+i)->next = (flow_temp+i+1);
      (tuple_temp+i)->next = (tuple_temp+i+1);
      (tuple_temp+i)->flow = NULL;
      (hash_table+i)->fivetuple = NULL;
    }
   (flow_temp+i)->next = NULL;
   (tuple_temp+i)->next = NULL;
   (tuple_temp+i)->flow = NULL;
   (hash_table+i)->fivetuple = NULL;
   
/*    flow_temp = free_flow_list; */
/*    tuple_temp = free_fivetuple_list; */
/*    i=0; */

/*    while (tuple_temp != NULL) */
/*      { */
/*        printf("Entry found %d\n",i); */
/*        tuple_temp = tuple_temp->next; */
/*        i++; */
/*      } */
/*    exit(0); */
}

// Hash function
// Modified version of function found in NetBSD code
unsigned int ipflow_hash(struct in_addr dst, struct in_addr src, unsigned short sport, unsigned short dport,unsigned char ip_p)
{
  unsigned int hash = ip_p;
  int idx;
  for (idx = 0; idx < 32; idx += IPFLOW_HASHBITS)
    hash += (dst.s_addr >> (32 - idx)) + (src.s_addr >> idx) + (sport >> idx) + (dport >> idx);
  return hash & (IPFLOW_HASHSIZE-1);
}

// To decode and print TCP flags
void decode_flags(unsigned char flags)
{
  if ((flags & 0x01) == 0x01) printf("FIN:");
  if ((flags & 0x02) == 0x02) printf("SYN:");
  if ((flags & 0x04) == 0x04) printf("RST:");
  if ((flags & 0x08) == 0x08) printf("PUSH:");
  if ((flags & 0x10) == 0x10) printf("ACK:");
  if ((flags & 0x01) == 0x20) printf("URG:");
}


int flowid(packet *pass_packet)
{  
  struct ip *iphdr;
  struct tcphdr *tcp;
  struct udphdr *udp;

  unsigned int hash;
  struct hash_entry *hashentry;
  struct fivetuple *curr_tuple;

  /*
   * Have we got an IP packet ?
   * If so, process packet, otherwise drop to trace file
   * Packets shouldn't be dropped with the TSH or DAG trace
   * files since the ll_type is fudged to 0x800
   */

  if (pass_packet->meta_buffer.ll_type != 0x800)
    {
#ifdef VERBOSE
      printf("\nNon-IP packet received, packet dropped");
      printf("\nType : 0x%x",pass_packet->meta_buffer.ll_type);
      printf("\nLength : %d",pass_packet->meta_buffer.ll_length);
#endif
      switch(trace_format)
	{
	case 0 : 
	  write_packet_to_tcpdump_file(pass_packet,DROP);
	  break;
	case 1 :
	  write_packet_to_tsh_file();
	  break;
	}	  
      return 1;
    }

  /*
   * Extract the IP header from the packet
   */
  iphdr = (struct ip*)(pass_packet->data);

  /*
   * Extract the TCP or UDP header depending on IP protocol
   * Compute hash appropriately. TCP and UDP use proper 5 tuple
   * "Other" packets use a 3 tuple with source and destination
   * ports as zero.
   */
  if (iphdr->ip_p == TCP_P)
    {
      tcp = (struct tcphdr*)((pass_packet->data)+IP_SIZE);
      hash = ipflow_hash(iphdr->ip_dst,iphdr->ip_src,tcp->th_sport,tcp->th_dport,iphdr->ip_p);
      tcp_count++;
      if (((tcp->th_flags)&0x01) == TH_FIN) numfins++;
      if (((tcp->th_flags)&0x02) == TH_SYN) numsyns++;
      if (((tcp->th_flags)&0x04) == TH_RST) numrsts++;
    }
  else if (iphdr->ip_p == UDP_P)
    {
      udp = (struct udphdr*)((pass_packet->data)+IP_SIZE);
      hash = ipflow_hash(iphdr->ip_dst,iphdr->ip_src,udp->uh_sport,udp->uh_dport,iphdr->ip_p);
      udp_count++;
    }
  else // "Other" packets, (not TCP or UDP)
    {
      hash = ipflow_hash(iphdr->ip_dst,iphdr->ip_src,0,0,iphdr->ip_p);
      other_count++;
    }
  
  // Look up the hash table
  hashentry = hash_table+hash;

  /* 
   * Case 1 : CREATE a new flow
   * at current hash location if it is NULL
   */
  if (hashentry->fivetuple == NULL)
    {
      if (free_fivetuple_list == NULL)
	{
	  printf("\n\n%s:%d(%s),Not enough memory to allocate 5 tuple. Free list is empty",__FILE__,__LINE__,__FUNCTION__);
	  exit(0);
	}
      else if (free_flow_list == NULL)
	{
	  printf("\n\n%s:%d(%s),Not enough memory to allocate flows. Free list is empty",__FILE__,__LINE__,__FUNCTION__);
	  exit(0); 
	}
      else
	{
	  hashentry->fivetuple = free_fivetuple_list;
	  hashentry->fivetuple->flow = free_flow_list;
	}
      free_fivetuple_list = free_fivetuple_list->next;
      free_flow_list = free_flow_list->next;
      hashentry->fivetuple->next = NULL;

      //copy the 5 tuple over
      hashentry->fivetuple->dst.s_addr = iphdr->ip_dst.s_addr;
      hashentry->fivetuple->src.s_addr = iphdr->ip_src.s_addr;
      hashentry->fivetuple->ip_p = iphdr->ip_p;
      if (iphdr->ip_p == TCP_P)
	{
	  hashentry->fivetuple->sport = tcp->th_sport;
	  hashentry->fivetuple->dport = tcp->th_dport;
	}
      else if (iphdr->ip_p == UDP_P)
	{
	  hashentry->fivetuple->sport = udp->uh_sport;
	  hashentry->fivetuple->dport = udp->uh_dport;
	}
      else // "other" packet set ports to arbit value
	{
	  hashentry->fivetuple->sport = 0;
	  hashentry->fivetuple->dport = 0;
	}

      // create flow record
      hashentry->fivetuple->flow->packet_count=0;
      hashentry->fivetuple->flow->packet_count++;
      hashentry->fivetuple->flow->ip_tos = iphdr->ip_tos;
      hashentry->fivetuple->flow->ip_off = iphdr->ip_off;
      hashentry->fivetuple->flow->packet_count_term=0;
      hashentry->fivetuple->flow->first_time = pass_packet->meta_buffer.time;
      hashentry->fivetuple->flow->last_time = pass_packet->meta_buffer.time;

      // We have to do different things depending on protocol
      if (iphdr->ip_p == TCP_P)
	{
	  // store initial seq,ack numbers and flags
	  hashentry->fivetuple->flow->stats.tcp_stats.th_firstseq = tcp->th_seq;
	  hashentry->fivetuple->flow->stats.tcp_stats.th_firstflags = tcp->th_flags;
	  hashentry->fivetuple->flow->stats.tcp_stats.th_firstack = tcp->th_ack;
	  // store final (last seen) seq,ack numbers,flags
	  // in case of active flows with one packet
	  hashentry->fivetuple->flow->stats.tcp_stats.th_lastseq = tcp->th_seq;
	  hashentry->fivetuple->flow->stats.tcp_stats.th_lastack = tcp->th_ack;
	  hashentry->fivetuple->flow->stats.tcp_stats.th_lastflags = tcp->th_flags;
	}
      else if (iphdr->ip_p == UDP_P)
	{
	  hashentry->fivetuple->flow->stats.udp_stats.byte_count=0;
	  hashentry->fivetuple->flow->stats.udp_stats.byte_count += udp->uh_ulen;
	}
      else // "other" packet
	{
	  // do nothing for now
	  // FIXME
	}

      // Check for flow termination
      // non TCP flows are all marked ACTIVE to begin with
      // TCP flows are checked for a termination condition

      hashentry->fivetuple->flow->status = TRUE; // (default value)

      if (iphdr->ip_p == TCP_P)
	{
	  if ((((tcp->th_flags)&0x04) == TH_RST) || (((tcp->th_flags)&0x01) == TH_FIN))
	    {
	      hashentry->fivetuple->flow->status = FALSE;
	    }
	}
    }
  /* 
   * END Case 1
   */

  /* 
   * Case 2 : Traverse linked list comparing 5/3 tuple
   * UPDATE record if it matches or CREATE new record at end
   * Repeat for different protocols
   */
  else
    {
      // store link to first pointer
      // traverse list using hashentry->fivetuple
      curr_tuple = hashentry->fivetuple;

      switch (iphdr->ip_p)
	{
	case TCP_P : // TCP protocol
	  while (hashentry->fivetuple != NULL)
	    {
	      // does the 5 tuple match ?
	      if ((hashentry->fivetuple->dst.s_addr == iphdr->ip_dst.s_addr) && (hashentry->fivetuple->src.s_addr == iphdr->ip_src.s_addr) && (hashentry->fivetuple->sport == tcp->th_sport) && (hashentry->fivetuple->dport == tcp->th_dport) && (hashentry->fivetuple->ip_p == iphdr->ip_p))
		{	      
		  // update flow record if flow is active
		  if (hashentry->fivetuple->flow->status == TRUE)
		    {
		      hashentry->fivetuple->flow->packet_count++;
		      hashentry->fivetuple->flow->ip_tos = iphdr->ip_tos;
		      hashentry->fivetuple->flow->ip_off = iphdr->ip_off;
		      hashentry->fivetuple->flow->last_time = pass_packet->meta_buffer.time;
		      // update final seq and ack nos.
		      hashentry->fivetuple->flow->stats.tcp_stats.th_lastseq = tcp->th_seq;
		      hashentry->fivetuple->flow->stats.tcp_stats.th_lastack = tcp->th_ack;
		      hashentry->fivetuple->flow->stats.tcp_stats.th_lastflags = tcp->th_flags;
		      // Check for flow termination
		      if ((((tcp->th_flags)&0x01) == TH_FIN) || (((tcp->th_flags)&0x04) == TH_RST))
			{
			  hashentry->fivetuple->flow->status = FALSE;
			}
		    }
		  else // packet hashes to a terminated flow
		    {
		      hashentry->fivetuple->flow->packet_count_term++;
		    }
		  break; // exit from while
		}
	      else if (hashentry->fivetuple->next == NULL)		
		// create new record @ end of the list
		{
		  collisons++;
		  if (free_fivetuple_list == NULL)
		    {
		      printf("\n\n%s:%d(%s),Not enough memory to allocate 5 tuple. Free list is empty",__FILE__,__LINE__,__FUNCTION__);
		      exit(0);
		    }
		  else if (free_flow_list == NULL)
		    {
		      printf("\n\n%s:%d(%s),Not enough memory to allocate flows. Free list is empty",__FILE__,__LINE__,__FUNCTION__);
		      exit(0); 
		    }
		  else
		    {
		      hashentry->fivetuple->next = free_fivetuple_list;
		      hashentry->fivetuple->next->flow = free_flow_list;
		    }
		  free_fivetuple_list = free_fivetuple_list->next;
		  free_flow_list = free_flow_list->next;
		  hashentry->fivetuple = hashentry->fivetuple->next;
		  hashentry->fivetuple->next = NULL;
		  
		  //copy the 5 tuple over
		  hashentry->fivetuple->dst.s_addr = iphdr->ip_dst.s_addr;
		  hashentry->fivetuple->src.s_addr = iphdr->ip_src.s_addr;
		  hashentry->fivetuple->sport = tcp->th_sport;
		  hashentry->fivetuple->dport = tcp->th_dport;
		  hashentry->fivetuple->ip_p = iphdr->ip_p;
	     
		  // create flow record
		  hashentry->fivetuple->flow->packet_count=0;
		  hashentry->fivetuple->flow->packet_count++;
		  hashentry->fivetuple->flow->ip_tos = iphdr->ip_tos;
		  hashentry->fivetuple->flow->ip_off = iphdr->ip_off;
		  hashentry->fivetuple->flow->packet_count_term=0;
		  hashentry->fivetuple->flow->first_time = pass_packet->meta_buffer.time;
		  hashentry->fivetuple->flow->last_time = pass_packet->meta_buffer.time;

		  // store initial seq, ack numbers and flags
		  hashentry->fivetuple->flow->stats.tcp_stats.th_firstseq = tcp->th_seq;
		  hashentry->fivetuple->flow->stats.tcp_stats.th_firstack = tcp->th_ack;
		  hashentry->fivetuple->flow->stats.tcp_stats.th_firstflags = tcp->th_flags;

		  // store final seq,ack numbers and flags
		  // in case of active flows with one packet
		  hashentry->fivetuple->flow->stats.tcp_stats.th_lastseq = tcp->th_seq;
		  hashentry->fivetuple->flow->stats.tcp_stats.th_lastack = tcp->th_ack;
		  hashentry->fivetuple->flow->stats.tcp_stats.th_lastflags = tcp->th_flags;

		  // Check for flow termination
		  hashentry->fivetuple->flow->status = TRUE; //(default value)

		  if ((((tcp->th_flags)&0x01) == TH_FIN) || (((tcp->th_flags)&0x04) == TH_RST))
		    {
		      hashentry->fivetuple->flow->status = FALSE;
		    }
		  break; // break from while
		}
	      else
		{
		  hashentry->fivetuple = hashentry->fivetuple->next;
		}
	    } // end while
	  break; // break switch

	case UDP_P : // UDP protocol
	  while (hashentry->fivetuple != NULL)
	    {
	      // does the 5 tuple match ?
	      if ((hashentry->fivetuple->dst.s_addr == iphdr->ip_dst.s_addr) && (hashentry->fivetuple->src.s_addr == iphdr->ip_src.s_addr) && (hashentry->fivetuple->sport == udp->uh_sport) && (hashentry->fivetuple->dport == udp->uh_dport) && (hashentry->fivetuple->ip_p == iphdr->ip_p))
		{	      
		  // update flow record if flow is active
		  if (hashentry->fivetuple->flow->status == TRUE)
		    {
		      hashentry->fivetuple->flow->packet_count++;
		      hashentry->fivetuple->flow->ip_tos = iphdr->ip_tos;
		      hashentry->fivetuple->flow->ip_off = iphdr->ip_off;
		      hashentry->fivetuple->flow->stats.udp_stats.byte_count += udp->uh_ulen;
		      // Check for flow termination
		      // Non TCP flows can terminate only
		      // via a timeout
		      if (((pass_packet->meta_buffer.time)-(hashentry->fivetuple->flow->last_time)) > TIMEOUT)
			{
			  hashentry->fivetuple->flow->status = FALSE;
			}
		      hashentry->fivetuple->flow->last_time = pass_packet->meta_buffer.time;
		    }
		  else // packet hashes to a terminated flow
		    {
		      hashentry->fivetuple->flow->packet_count_term++;
		    }
		  break; // exit from while
		}
	      else if (hashentry->fivetuple->next == NULL)
		// create new record @ end of the list
		{
		  collisons++;
		  if (free_fivetuple_list == NULL)
		    {
		      printf("\n\n%s:%d(%s),Not enough memory to allocate 5 tuple. Free list is empty",__FILE__,__LINE__,__FUNCTION__);
		      exit(0);
		    }
		  else if (free_flow_list == NULL)
		    {
		      printf("\n\n%s:%d(%s),Not enough memory to allocate flows. Free list is empty",__FILE__,__LINE__,__FUNCTION__);
		      exit(0); 
		    }
		  else
		    {
		      hashentry->fivetuple->next = free_fivetuple_list;
		      hashentry->fivetuple->next->flow = free_flow_list;
		    }
		  free_fivetuple_list = free_fivetuple_list->next;
		  free_flow_list = free_flow_list->next;
		  hashentry->fivetuple = hashentry->fivetuple->next;
		  hashentry->fivetuple->next = NULL;
		  
		  //copy the 5 tuple over
		  hashentry->fivetuple->dst.s_addr = iphdr->ip_dst.s_addr;
		  hashentry->fivetuple->src.s_addr = iphdr->ip_src.s_addr;
		  hashentry->fivetuple->sport = udp->uh_sport;
		  hashentry->fivetuple->dport = udp->uh_dport;
		  hashentry->fivetuple->ip_p = iphdr->ip_p;
	     
		  // create flow record
		  hashentry->fivetuple->flow->packet_count=0;
		  hashentry->fivetuple->flow->packet_count++;
		  hashentry->fivetuple->flow->ip_tos = iphdr->ip_tos;
		  hashentry->fivetuple->flow->ip_off = iphdr->ip_off;
		  hashentry->fivetuple->flow->packet_count_term=0;
		  hashentry->fivetuple->flow->stats.udp_stats.byte_count = 0;
		  hashentry->fivetuple->flow->stats.udp_stats.byte_count += udp->uh_ulen;
		  hashentry->fivetuple->flow->first_time = pass_packet->meta_buffer.time;
		  hashentry->fivetuple->flow->last_time = pass_packet->meta_buffer.time;

		  // NON TCP FLOWS ARE ACTIVE TO START WITH
		  hashentry->fivetuple->flow->status = TRUE;
		  break; // break from while
		}
	      else
		{
		  hashentry->fivetuple = hashentry->fivetuple->next;
		}
	    } // end while
	  break; // break switch

	default : // Other protocols
	  while (hashentry->fivetuple != NULL)
	    {
	      // does the 3 tuple match ?
	      if ((hashentry->fivetuple->dst.s_addr == iphdr->ip_dst.s_addr) && (hashentry->fivetuple->src.s_addr == iphdr->ip_src.s_addr) && (hashentry->fivetuple->ip_p == iphdr->ip_p))
		{	      
		  // update flow record if flow is active
		  if (hashentry->fivetuple->flow->status == TRUE)
		    {
		      hashentry->fivetuple->flow->packet_count++;
		      hashentry->fivetuple->flow->ip_tos = iphdr->ip_tos;
		      hashentry->fivetuple->flow->ip_off = iphdr->ip_off;
		      // Check for flow termination
		      // Non TCP flows can terminate only
		      // via a timeout
		      if (((pass_packet->meta_buffer.time)-(hashentry->fivetuple->flow->last_time)) > TIMEOUT)
			{
			  hashentry->fivetuple->flow->status = FALSE;
			}
		      hashentry->fivetuple->flow->last_time = pass_packet->meta_buffer.time;
		    }
		  else // packet hashes to a terminated flow
		    {
		      hashentry->fivetuple->flow->packet_count_term++;
		    }
		  break; // exit from while
		}
	      else if (hashentry->fivetuple->next == NULL)
		// create new record @ end of the list
		{
		  collisons++;
		  if (free_fivetuple_list == NULL)
		    {
		      printf("\n\n%s:%d(%s),Not enough memory to allocate 5 tuple. Free list is empty",__FILE__,__LINE__,__FUNCTION__);
		      exit(0);
		    }
		  else if (free_flow_list == NULL)
		    {
		      printf("\n\n%s:%d(%s),Not enough memory to allocate flows. Free list is empty",__FILE__,__LINE__,__FUNCTION__);
		      exit(0); 
		    }
		  else
		    {
		      hashentry->fivetuple->next = free_fivetuple_list;
		      hashentry->fivetuple->next->flow = free_flow_list;
		    }
		  free_fivetuple_list = free_fivetuple_list->next;
		  free_flow_list = free_flow_list->next;
		  hashentry->fivetuple = hashentry->fivetuple->next;
		  hashentry->fivetuple->next = NULL;
		  
		  //copy the 5 tuple over
		  hashentry->fivetuple->dst.s_addr = iphdr->ip_dst.s_addr;
		  hashentry->fivetuple->src.s_addr = iphdr->ip_src.s_addr;
		  hashentry->fivetuple->sport = 0;
		  hashentry->fivetuple->dport = 0;
		  hashentry->fivetuple->ip_p = iphdr->ip_p;
	     
		  // create flow record
		  hashentry->fivetuple->flow->packet_count=0;
		  hashentry->fivetuple->flow->packet_count++;
		  hashentry->fivetuple->flow->ip_tos = iphdr->ip_tos;
		  hashentry->fivetuple->flow->ip_off = iphdr->ip_off;
		  hashentry->fivetuple->flow->packet_count_term=0;
		  hashentry->fivetuple->flow->first_time = pass_packet->meta_buffer.time;
		  hashentry->fivetuple->flow->last_time = pass_packet->meta_buffer.time;

		  // NON TCP FLOWS ARE ACTIVE TO START WITH
		  hashentry->fivetuple->flow->status = TRUE;
		  break; // break from while
		}
	      else
		{
		  hashentry->fivetuple = hashentry->fivetuple->next;
		}
	    } // end while
	  break; // break switch

	} // end switch

      // restore link for first pointer
      hashentry->fivetuple = curr_tuple;
    }
  /*
   * END Case 2
   */


  /*
   * Function will return 0 if it needs to store the packet in queue
   * Queue pointers are modified based on return value -> ? FIXME ?
   */
  
  switch(trace_format)
    {
    case 0 : 
      write_packet_to_tcpdump_file(pass_packet,DUMP);
      break;
    case 1 :
      write_packet_to_tsh_file();
      break;
    }
  
  return 1;
}

// Print flow statistics
void print_flow()
{
  int i,tp=0,fc=0;
  int af=0,iaf=0,tterm=0;

  for (i=0;i<IPFLOW_HASHSIZE;i++)
    {
      while ((hash_table+i)->fivetuple != NULL)
	{
/* 	  printf("\n(%-8x,%-8x,%-5d,%-5d,%-3d) -> %d\tpackets (hashes to %d)",ntohl((hash_table+i)->fivetuple->src.s_addr),ntohl((hash_table+i)->fivetuple->dst.s_addr),ntohs((hash_table+i)->fivetuple->sport),ntohs((hash_table+i)->fivetuple->dport),(hash_table+i)->fivetuple->ip_p,(hash_table+i)->fivetuple->flow->packet_count,i); */
/* 	  if ((hash_table+i)->fivetuple->ip_p == TCP_P) */
/* 	    { */
/* 	      printf("\nTCP Flow:INITIAL (SEQ,ACK,FLAGS) --> (%8x,",ntohl((hash_table+i)->fivetuple->flow->stats.tcp_stats.th_firstseq)); */
/* 	      ((((hash_table+i)->fivetuple->flow->stats.tcp_stats.th_firstflags)&0x10) == TH_ACK) ? printf("%8x",ntohl((hash_table+i)->fivetuple->flow->stats.tcp_stats.th_firstack)) : printf("00dead00"); */
/* 	      printf(", 0x%2x - ",(hash_table+i)->fivetuple->flow->stats.tcp_stats.th_firstflags); */
/* 	      decode_flags((hash_table+i)->fivetuple->flow->stats.tcp_stats.th_firstflags); */
/* 	      printf(")"); */

/* 	      printf("\nTCP Flow:FINAL   (SEQ,ACK,FLAGS) --> (%8x,",ntohl((hash_table+i)->fivetuple->flow->stats.tcp_stats.th_lastseq)); */
/* 	      ((((hash_table+i)->fivetuple->flow->stats.tcp_stats.th_lastflags)&0x10) == TH_ACK) ? printf("%8x",ntohl((hash_table+i)->fivetuple->flow->stats.tcp_stats.th_lastack)) : printf("00dead00"); */
/* 	      printf(", 0x%2x - ",(hash_table+i)->fivetuple->flow->stats.tcp_stats.th_lastflags); */
/* 	      decode_flags((hash_table+i)->fivetuple->flow->stats.tcp_stats.th_lastflags); */
/* 	      printf(")"); */

/* 	    } */
/* 	  else if ((hash_table+i)->fivetuple->ip_p == UDP_P) */
/* 	    { */
/* 	      printf("\nUDP Flow:BYTE COUNT : %d",(hash_table+i)->fivetuple->flow->stats.udp_stats.byte_count); */
/* 	    } */
/* 	  else */
/* 	    { */
/* 	      printf("\nOTHER flow"); */
/* 	    } */
/* 	  printf("\nSTATUS --> "); */
	  
/* 	  if (((hash_table+i)->fivetuple->flow->status) == TRUE) printf("ACTIVE"); */
/* 	  if (((hash_table+i)->fivetuple->flow->status) == FALSE) printf("IN-ACTIVE"); */
/* 	  printf(" ,Packets after termination = %d\n",(hash_table+i)->fivetuple->flow->packet_count_term); */
/* 	  printf("First packet seen at %f\n",(hash_table+i)->fivetuple->flow->first_time); */
/* 	  printf("Last packet seen at %f\n",(hash_table+i)->fivetuple->flow->last_time); */
/* 	  printf("Flow duration : %f sec\n",((hash_table+i)->fivetuple->flow->last_time)-((hash_table+i)->fivetuple->flow->first_time)); */
	  
	  tterm+=(hash_table+i)->fivetuple->flow->packet_count_term;
	  if ((hash_table+i)->fivetuple->flow->status == TRUE) 
	    af++;
	  else if ((hash_table+i)->fivetuple->flow->status == FALSE)
	    iaf++;
	  tp+=(hash_table+i)->fivetuple->flow->packet_count;
	  fc++;
	  (hash_table+i)->fivetuple = (hash_table+i)->fivetuple->next;
	}
    }
#ifdef GLOBAL_PACKET_MEMORY
  printf("\nUsing GLOBAL packet memory");
#else
  printf("\nUsing LOCAL packet memory");
#endif
  printf("\nTotal number of flows = %d",fc);
  printf("\nActive flows = %d",af);
  printf("\nInactive flows = %d",iaf);
  printf("\nTotal number of packets processed = %d\n",tp);
  printf("\nPackets after flow termination = %d\n",tterm);
  printf("\nTotal SYNS = %d",numsyns);
  printf("\nTotal FINS = %d",numfins);
  printf("\nTotal RST = %d",numrsts);
  printf("\nTCP packets = %d",tcp_count);
  printf("\nUDP packets = %d",udp_count);
  printf("\nOther packets = %d",other_count);
  printf("\nCollisons = %d\n",collisons);
}

