/*
    PacketBench - Workload Characterization Tool
    Copyright (C) 2004  Ramaswamy Ramaswamy

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "bench.h"
#include "flowid.h"

#ifdef GLOBAL_PACKET_MEMORY
packet top_packet[PACKET_QUEUE_SIZE];
packet *process_packet=&top_packet[0];
#endif

/* 
 * trace file descriptors for processed and dropped packets
 */
static pcap_dumper_t *dump_file;
static pcap_dumper_t *drop_file;

/*
 *globals for storing time, and packet statistics
 */
static struct timeval global_time;
static int total_packets,dumped_packets,dropped_packets;

/*
 * Globals to store values of various header sizes
 */
static int PKTHDR_SIZE;
static int ETHER_SIZE;
static int IP_SIZE;
static int TCP_SIZE;

void write_packet_to_tsh_file()
{
}

/*
 * Function writes packets to either a dump file or a
 * drop file depending on the value of action.
 * Ethernet and pcap headers are fudged.
 */
void write_packet_to_tcpdump_file(packet *packet_pointer,int action)
{
  struct pcap_pkthdr pkthdr;
  struct ether_header eth;
  unsigned char *dump_packet;

  /*
   * Fudge the pcap packet header
   * Time is initalized to current system time
   * + INCTIMEUNITS per packet
   */
  pkthdr.ts.tv_sec = global_time.tv_sec;
  global_time.tv_usec += INCTIMEUNITS;
  pkthdr.ts.tv_usec = global_time.tv_usec;

  /*
   * Copy packet length from meta buffer
   */
  pkthdr.caplen = (bpf_u_int32)packet_pointer->meta_buffer.ll_length;
  pkthdr.len = (bpf_u_int32)packet_pointer->meta_buffer.ll_length;

  /*
   * Fudge ethernet header
   * Source and dest. addr are fake
   * Type is copied from meta buffer
   */
  if (action == DUMP)
    {
      eth.ether_dhost[0]=(unsigned char)0xDE;
      eth.ether_dhost[1]=(unsigned char)0xAD;
      eth.ether_dhost[2]=(unsigned char)0x5C;
      eth.ether_dhost[3]=(unsigned char)0xA1;
      eth.ether_dhost[4]=(unsigned char)0xAB;
      eth.ether_dhost[5]=(unsigned char)0x1E;

      eth.ether_shost[0]=(unsigned char)0xDE;
      eth.ether_shost[1]=(unsigned char)0xAD;
      eth.ether_shost[2]=(unsigned char)0x5C;
      eth.ether_shost[3]=(unsigned char)0xA1;
      eth.ether_shost[4]=(unsigned char)0xAB;
      eth.ether_shost[5]=(unsigned char)0x1E;
    }

  if (action == DROP)
    {
      eth.ether_dhost[0]=(unsigned char)0xDE;
      eth.ether_dhost[1]=(unsigned char)0xAD;
      eth.ether_dhost[2]=(unsigned char)0xDE;
      eth.ether_dhost[3]=(unsigned char)0xAD;
      eth.ether_dhost[4]=(unsigned char)0xDE;
      eth.ether_dhost[5]=(unsigned char)0xAD;

      eth.ether_shost[0]=(unsigned char)0xDE;
      eth.ether_shost[1]=(unsigned char)0xAD;
      eth.ether_shost[2]=(unsigned char)0xDE;
      eth.ether_shost[3]=(unsigned char)0xAD;
      eth.ether_shost[4]=(unsigned char)0xDE;
      eth.ether_shost[5]=(unsigned char)0xAD;
    }
      
  eth.ether_type = (u_int16_t)(htons((uint16_t)packet_pointer->meta_buffer.ll_type));

  /*
   * Create a packet with our fudged ethernet header in front
   */
  dump_packet=malloc((size_t)packet_pointer->meta_buffer.ll_length);

  if (dump_packet == NULL)
    {
      printf("\n%s:%d(%s),Unable to allocate memory for packet",__FILE__,__LINE__,__FUNCTION__);
      exit(0);
    }

  memcpy(dump_packet,&eth,(size_t)ETHER_SIZE);
  memcpy(dump_packet+ETHER_SIZE,packet_pointer->data,(size_t)(packet_pointer->meta_buffer.ll_length-ETHER_SIZE));

  /*
   * Write packet back to appropriate trace file
   * Free memory used, update statistics
   */
  if (action == DUMP)
    {
#ifdef VERBOSE
      printf("\n%s:%d(%s),Writing packet to dump file",__FILE__,__LINE__,__FUNCTION__);
#endif
      pcap_dump((u_char *)dump_file,&pkthdr,dump_packet);
      dumped_packets++;
    }
  
  else if (action == DROP)
    {
#ifdef VERBOSE
      printf("\n%s:%d(%s),Writing packet to drop file",__FILE__,__LINE__,__FUNCTION__);
#endif
      pcap_dump((u_char *)drop_file,&pkthdr,dump_packet);
      dropped_packets++;
    }

  else
    {
      printf("\n%s:%d(%s),Undefined action to handle packet",__FILE__,__LINE__,__FUNCTION__);
    }
  free(dump_packet);
}

/*
 * Initalize the packet queue.
 * Function returns a pointer to a block of allocated
 * memory which can store PACKET_QUEUE_SIZE packets
 */

static packet * init_packet_queue()
{
  int i;

  packet *packet_queue,*temp;

  packet_queue = malloc(PACKET_QUEUE_SIZE*PACKET_SIZE);
  
  if (packet_queue == NULL)
    {
      printf("\n%s:%d(%s),Unable to allocate memory for packet queue",__FILE__,__LINE__,__FUNCTION__);
      exit(0);
    }
  
#ifdef VERBOSE
  printf("\n\n%s:%d(%s),Allocated memory for packet queue",__FILE__,__LINE__,__FUNCTION__);
#endif

  temp = packet_queue;

  /*
   * Link up the PACKET_QUEUE_SIZE remaining locations
   */
  for(i=0;i<PACKET_QUEUE_SIZE-1;i++)
    {
      (temp+i)->meta_buffer.next = (temp+i+1);
    }
  
   (temp+i)->meta_buffer.next = NULL;

  return packet_queue;
}

void usage()
{
  printf("\nUsage : bench {options} {tracefile} {dumpfile} {dropfile}\n");
  printf("Options:\n");
  printf("-N : trace file is in TSH format (NLANR)\n");
  printf("-T : trace file is in tcpdump format\n");
  exit(0);
}

/*************************************************** 
 *                 MAIN FUNCTION                   *
 ***************************************************/

int main(int argc, char *argv[])
{
  pcap_t *trace_descr; /* For storing trace file/interface info */
  char errbuf[PCAP_ERRBUF_SIZE]; /* for storing error messages */
  struct pcap_pkthdr pkthdr; /* required by pcap */
  const u_char *captured_packet; /* packet is stored here */
  struct ether_header ether_header; /* store ether info per packet */
  char *format;
  FILE *tsh_descr;
  unsigned char tsh_buffer[TSH_RECORD_SIZE];
  unsigned int ts_sec,ts_usec;

#ifndef GLOBAL_PACKET_MEMORY
  /*
   * Pointers to the packet queue. Point to top of queue
   * and the next packet that needs to be processed
   */
  packet *top_packet,*process_packet;
#endif

  /*
   * Function pointer to benchmark function
   */
  int (*process_packet_function)(packet *);

  /*
   * RAW tcpdump tracefile pointers
   * Values initalized from command line
   * dropfile is /dev/null if not specified
   */
  char *tracefile;
  char *dumpfile;
  char *dropfile;

  /*
   * Parse command line args
   */
  switch (argc)
    {
    case 5 :
      format = argv[1];
      tracefile = argv[2];
      dumpfile = argv[3];
      dropfile = argv[4];
      break;

    default :
      usage();
    }
  
  if (strcmp(format,"-N") == 0) 
    {
      trace_format = 1;
    }
  else if (strcmp(format,"-T") == 0) 
    {
      trace_format = 0;
    }
  else 
    {
      printf("\nUnknown option %s",format);
      usage();
    }

#ifdef VERBOSE
  switch(trace_format)
    {
    case 0 : 
      printf("\nUsing TCPDUMP tracefile"); 
      break;
    case 1 : 
      printf("\nUsing TSH tracefile"); 
      break;
    }
  printf("\nInput trace file : %s",tracefile);
  printf("\nDump file : %s",dumpfile);
  printf("\nDrop file : %s",dropfile);
#endif

  /*
   * Initialize packet sizes
   */
  PKTHDR_SIZE = (int)sizeof(struct pcap_pkthdr);
  ETHER_SIZE = (int)sizeof(struct ether_header);
  IP_SIZE = (int)sizeof(struct ip);
  TCP_SIZE = (int)sizeof(struct tcphdr);

#ifdef VERBOSE
  printf("\n\nSize of packet header header = %d bytes",PKTHDR_SIZE);
  printf("\nSize of ethernet header = %d bytes",ETHER_SIZE);
  printf("\nSize of IP header = %d bytes",IP_SIZE);
  printf("\nSize of TCP header = %d bytes",TCP_SIZE);
#endif

#ifndef GLOBAL_PACKET_MEMORY
#ifdef VERBOSE
  printf("\nUsing LOCAL packet memory");
#endif
  /*
   * Initialize the packet queue
   * top_packet always points to top of packet queue
   */
  top_packet = init_packet_queue();
  process_packet = top_packet;
#endif

#ifdef GLOBAL_PACKET_MEMORY
#ifdef VERBOSE
  printf("\nUsing GLOBAL packet memory");
#endif
#endif

  /*
   * Make function pointer point to benchmark function
   * BENCHMARK_FUNCTION has to be defined in Makefile
   */
  process_packet_function = &BENCHMARK_FUNCTION;

  /*
   * Initialize the data structures needed for flow-id
   */
  init_flowid();
  
  /* 
   * Open the dumpfiles
   */
  switch(trace_format)
    {
    case 0 : 
      trace_descr = pcap_open_offline(tracefile, errbuf); 
      dump_file=pcap_dump_open(trace_descr,dumpfile);
      drop_file=pcap_dump_open(trace_descr,dropfile);
      if((trace_descr == NULL)||(dump_file == NULL)||(drop_file == NULL))
	{
	  printf("\n%s:%d(%s),Cannot open dumpfile: %s",__FILE__,__LINE__,__FUNCTION__,errbuf);
	  exit(0);
	}
      break;
    case 1 : 
      tsh_descr = fopen(tracefile,"r"); 
      if (tsh_descr == NULL)
	{
	  printf("\n%s:%d(%s),Cannot open tracefile",__FILE__,__LINE__,__FUNCTION__);
	  exit(0);
	}
      break;
    }
  
  /* 
   * Initialize time of day for writing back to packets
   */
  gettimeofday(&global_time, NULL);

  switch(trace_format)
    {
    case 0 : 
      captured_packet=pcap_next(trace_descr,&pkthdr); 
      break;
    case 1 : 
      captured_packet=tsh_buffer;
      fread(tsh_buffer,TSH_RECORD_SIZE,1,tsh_descr); 
      break;
    }

  /**********************************************
   * For each packet returned by pcap, main loop*
   **********************************************/
  while (captured_packet != NULL)
    { 
      total_packets++;
	
      switch(trace_format)
	{
	case 0 :  // TCPDUMP
	  /* 
	   * Store packet contents from IP header in packet queue
	   */
	  memcpy(process_packet->data,(captured_packet+ETHER_SIZE),(size_t)(pkthdr.len-ETHER_SIZE));
	  
	  /*
	   * Extract link layer length and type from pcap header and ethernet 
	   * header
	   * Update meta buffer contents
	   */
	  memcpy(&ether_header,(struct ether_header *)(captured_packet),(size_t)ETHER_SIZE);
	  process_packet->meta_buffer.ll_length = (int)pkthdr.len;
	  process_packet->meta_buffer.ll_type = (unsigned short)ntohs((uint16_t)ether_header.ether_type);
	  /*
	   * Perform a sanity check to see if the packet payload
	   * is present. If not, print a warning message
	   */
	  if (pkthdr.caplen < pkthdr.len)
	    {
#ifdef VERBOSE
	      printf("\n%s:%d(%s), Warning : Packet payload possibly missing (caplen < len)",__FILE__,__LINE__,__FUNCTION__);
#endif
	    }
	  break;

	case 1 :   // NLANR TSH
	  /*
	   * Things to note here
	   * 1. TCP header is incomplete
	   * 2. Packet payload contains junk
	   * 3. The process packet function will operate
	   *    on ALL packets irrespective of the LL type
	   *    (we dont have link layer type in TSH)
	   * 4. NO DATA > LAYER 4
	   */
	  memcpy(process_packet->data,(captured_packet+8),TSH_RECORD_SIZE-8);

	  // Copy the timestamp
	  memcpy(&ts_sec,captured_packet,4);
	  memcpy(&ts_usec,captured_packet+4,4); 

	  // Convert to host byte order and mask out interface number
	  ts_sec=ntohl(ts_sec);
	  ts_usec=ntohl(ts_usec);	  
	  ts_usec &= 0x00ffffff;	  

	  // Store timestamp in meta buffer
	  process_packet->meta_buffer.time = (double)(ts_usec/1e6)+(double)(ts_sec);
	  process_packet->meta_buffer.ll_length = 999; //fake
	  process_packet->meta_buffer.ll_type = 0x800; // spoof to be IP

#ifdef VERBOSE
	    printf("\n%s:%d(%s), Warning : TSH trace file used. Packet payload is missing",__FILE__,__LINE__,__FUNCTION__);
#endif
	  break;
	}
      
      /*
       * Process the packet, (do Ipv4 routing in this case)
       * if process function decides to store packet in queue,
       * updated queue pointer to reflect next free location
       * No bounds checking is done! in case > 64 packets are
       * stored in queue -> ? FIXME ?
       */
      
#ifdef VERBOSE
      printf("\n\n%s:%d(%s),Processing packet...",__FILE__,__LINE__,__FUNCTION__);
#endif

      if (!((*process_packet_function)(process_packet)))
	{
	  process_packet = process_packet->meta_buffer.next;
	}
     
      switch(trace_format)
	{
	case 0 : 
	  captured_packet=pcap_next(trace_descr,&pkthdr); 
	  break;
	case 1 : // force loop to exit if we cant read from TSH file
	  if (fread(tsh_buffer,TSH_RECORD_SIZE,1,tsh_descr)==0)
	    captured_packet = NULL;	  
	  break;
	}
    }
      
#ifdef VERBOSE
  printf("\n\nTotal packets processed : %d",total_packets);
  printf("\nPackets dumped          : %d",dumped_packets);
  printf("\nPackets dropped         : %d\n",dropped_packets);
#endif

  switch(trace_format)
    {
    case 0 :
      pcap_dump_close(dump_file);
      pcap_dump_close(drop_file);
      break;
    case 1 :
      break;
    }

#ifdef VERBOSE
  print_flow();
#endif

#ifndef GLOBAL_PACKET_MEMORY
  free(top_packet);
#endif

  return 0;
}

