/* Radix-Tree Routing
 * by Srinivasan Venkatachary, modified by Tilman Wolf 1999
 * based on the radix-tree routing code from the NetBSD kernel 1.3.2
 * USE AT OWN RISK!
 */


#include <stdlib.h>
#include <sys/time.h>
#include <sys/resource.h>

#include "rourad.h"

extern char *optarg;

int	max_keylen;
struct radix_mask *rn_mkfreelist;
struct radix_node_head *mask_rnhead;
static char *addmask_key;
static char normal_chars[] = {0, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfc, 0xfe, -1};
static char *rn_zeros, *rn_ones;

#define rn_masktop (mask_rnhead->rnh_treetop)
#undef Bcmp
#define Bcmp(a, b, l) (l == 0 ? 0 : bcmp((caddr_t)(a), (caddr_t)(b), (u_long)l))

#define log(a,b) puts(b)
#define log5(a,b,c,d,e) printf(b,c,d,e)

unsigned int memUsed=0;
unsigned int memForTree=0;

/*NBYTES is the number of bytes in the address*/

static int rn_satsifies_leaf (char *, struct radix_node *, int);
static int rn_lexobetter (void *, void *);
static struct radix_mask *rn_new_radix_mask (struct radix_node *,
						 struct radix_mask *);

static void p_rtnode(struct radix_node *rn);

/*
 * The data structure for the keys is a radix tree with one way
 * branching removed.  The index rn_b at an internal node n represents a bit
 * position to be tested.  The tree is arranged so that all descendants
 * of a node n have keys whose bits all agree up to position rn_b - 1.
 * (We say the index of n is rn_b.)
 *
 * There is at least one descendant which has a one bit at position rn_b,
 * and at least one with a zero there.
 *
 * A route is determined by a pair of key and mask.  We require that the
 * bit-wise logical and of the key and mask to be the key.
 * We define the index of a route to associated with the mask to be
 * the first bit number in the mask where 0 occurs (with bit number 0
 * representing the highest order bit).
 *
 * We say a mask is normal if every bit is 0, past the index of the mask.
 * If a node n has a descendant (k, m) with index(m) == index(n) == rn_b,
 * and m is a normal mask, then the route applies to every descendant of n.
 * If the index(m) < rn_b, this implies the trailing last few bits of k
 * before bit b are all 0, (and hence consequently true of every descendant
 * of n), so the route applies to all descendants of the node as well.
 *
 * Similar logic shows that a non-normal mask m such that
 * index(m) <= index(n) could potentially apply to many children of n.
 * Thus, for each non-host route, we attach its mask to a list at an internal
 * node as high in the tree as we can go.
 *
 * The present version of the code makes use of normal routes in short-
 * circuiting an explict mask and compare operation when testing whether
 * a key satisfies a normal route, and also in remembering the unique leaf
 * that governs a subtree.
 */


#define NTIMES 1000000


struct radix_node *
rn_addmask(void *n_arg, int search, int skip);

struct radix_node *
rn_match(void *v_arg, struct radix_node_head *head);

struct radix_node *
rn_search(v_arg, head)
	void *v_arg;
	struct radix_node *head;
{
	register struct radix_node *x;
	register caddr_t v;

	for (x = head, v = v_arg; x->rn_b >= 0;) {
		if (x->rn_bmask & v[x->rn_off])
			x = x->rn_r;
		else
			x = x->rn_l;
	}
	return (x);
};

struct radix_node *
rn_search_m(v_arg, head, m_arg)
	struct radix_node *head;
	void *v_arg, *m_arg;
{
	register struct radix_node *x;
	register caddr_t v = v_arg, m = m_arg;


	for (x = head; x->rn_b >= 0;) {
		if ((x->rn_bmask & m[x->rn_off]) &&
		    (x->rn_bmask & v[x->rn_off]))
			x = x->rn_r;
		else
			x = x->rn_l;
	}
	return x;
};

int
rn_refines(m_arg, n_arg)
	void *m_arg, *n_arg;
{
	register caddr_t m = m_arg, n = n_arg;
	register caddr_t lim, lim2 = lim = n + *(u_char *)n;
	int longer = (*(u_char *)n++) - (int)(*(u_char *)m++);
	int masks_are_equal = 1;

	if (longer > 0)
		lim -= longer;
	while (n < lim) {
		if (*n & ~(*m))
			return 0;
		if (*n++ != *m++)
			masks_are_equal = 0;
	}
	while (n < lim2)
		if (*n++)
			return 0;
	if (masks_are_equal && (longer < 0))
		for (lim2 = m - longer; m < lim2; )
			if (*m++)
				return 1;
	return (!masks_are_equal);
}

struct radix_node *
rn_lookup(v_arg, m_arg, head)
	void *v_arg, *m_arg;
	struct radix_node_head *head;
{
	register struct radix_node *x;
	caddr_t netmask = 0;

	if (m_arg) {
		if ((x = rn_addmask(m_arg, 1, head->rnh_treetop->rn_off)) == 0)
			return (0);
		netmask = x->rn_key;
	}
	x = rn_match(v_arg, head);
	if (x && netmask) {
		while (x && x->rn_mask != netmask)
			x = x->rn_dupedkey;
	}
	return x;
}

#define min(x,y) (( (x) < (y) )? (x) : (y))

static int
rn_satsifies_leaf(trial, leaf, skip)
	char *trial;
	register struct radix_node *leaf;
	int skip;
{
	register char *cp = trial, *cp2 = leaf->rn_key, *cp3 = leaf->rn_mask;
	char *cplim;
	int length = min(*(u_char *)cp, *(u_char *)cp2);

	if (cp3 == 0)
		cp3 = rn_ones;
	else
		length = min(length, *(u_char *)cp3);
	cplim = cp + length; cp3 += skip; cp2 += skip;
	for (cp += skip; cp < cplim; cp++, cp2++, cp3++)
		if ((*cp ^ *cp2) & *cp3)
			return 0;
	return 1;
}

struct radix_node *
rn_match(v_arg, head)
	void *v_arg;
	struct radix_node_head *head;
{
	caddr_t v = v_arg;
	register struct radix_node *t = head->rnh_treetop, *x;
	register caddr_t cp = v, cp2;
	caddr_t cplim;
	struct radix_node *saved_t, *top = t;
	int off = t->rn_off, vlen = *(u_char *)cp, matched_off;
	register int test, b, rn_b;

	/*
	 * Open code rn_search(v, top) to avoid overhead of extra
	 * subroutine call.
	 */
	for (; t->rn_b >= 0; ) {
		if (t->rn_bmask & cp[t->rn_off])
			t = t->rn_r;
		else
			t = t->rn_l;
	}
	/*
	 * See if we match exactly as a host destination
	 * or at least learn how many bits match, for normal mask finesse.
	 *
	 * It doesn't hurt us to limit how many bytes to check
	 * to the length of the mask, since if it matches we had a genuine
	 * match and the leaf we have is the most specific one anyway;
	 * if it didn't match with a shorter length it would fail
	 * with a long one.  This wins big for class B&C netmasks which
	 * are probably the most common case...
	 */
	if (t->rn_mask)
		vlen = *(u_char *)t->rn_mask;
	cp += off; cp2 = t->rn_key + off; cplim = v + vlen;
	for (; cp < cplim; cp++, cp2++)
		if (*cp != *cp2)
			goto on1;
	/*
	 * This extra grot is in case we are explicitly asked
	 * to look up the default.  Ugh!
	 */
	if ((t->rn_flags & RNF_ROOT) && t->rn_dupedkey)
		t = t->rn_dupedkey;
	return t;
on1:
	test = (*cp ^ *cp2) & 0xff; /* find first bit that differs */
	for (b = 7; (test >>= 1) > 0;)
		b--;
	matched_off = cp - v;
	b += matched_off << 3;
	rn_b = -1 - b;
	/*
	 * If there is a host route in a duped-key chain, it will be first.
	 */
	if ((saved_t = t)->rn_mask == 0)
		t = t->rn_dupedkey;
	for (; t; t = t->rn_dupedkey)
		/*
		 * Even if we don't match exactly as a host,
		 * we may match if the leaf we wound up at is
		 * a route to a net.
		 */
		if (t->rn_flags & RNF_NORMAL) {
			if (rn_b <= t->rn_b)
				return t;
		} else if (rn_satsifies_leaf(v, t, matched_off))
				return t;
	t = saved_t;
	/* start searching up the tree */
	do {
		register struct radix_mask *m;
		t = t->rn_p;
		if ((m = t->rn_mklist) != NULL) {
			/*
			 * If non-contiguous masks ever become important
			 * we can restore the masking and open coding of
			 * the search and satisfaction test and put the
			 * calculation of "off" back before the "do".
			 */
			do {
				if (m->rm_flags & RNF_NORMAL) {
					if (rn_b <= m->rm_b)
						return (m->rm_leaf);
				} else {
					off = min(t->rn_off, matched_off);
					x = rn_search_m(v, t, m->rm_mask);
					while (x && x->rn_mask != m->rm_mask)
						x = x->rn_dupedkey;
					if (x && rn_satsifies_leaf(v, x, off))
						    return x;
				}
			} while ((m = m->rm_mklist) != NULL);
		}
	} while (t != top);
	return 0;
};

#ifdef RN_DEBUG
int	rn_nodenum;
struct	radix_node *rn_clist;
int	rn_saveinfo;
int	rn_debug =  1;
#endif

struct radix_node *
rn_newpair(v, b, nodes)
	void *v;
	int b;
	struct radix_node nodes[2];
{
	register struct radix_node *tt = nodes, *t = tt + 1;
	t->rn_b = b; t->rn_bmask = 0x80 >> (b & 7);
	t->rn_l = tt; t->rn_off = b >> 3;
	tt->rn_b = -1; tt->rn_key = (caddr_t)v; tt->rn_p = t;
	tt->rn_flags = t->rn_flags = RNF_ACTIVE;
#ifdef RN_DEBUG
	tt->rn_info = rn_nodenum++; t->rn_info = rn_nodenum++;
	tt->rn_twin = t; tt->rn_ybro = rn_clist; rn_clist = tt;
#endif
	return t;
}

struct radix_node *
rn_insert(v_arg, head, dupentry, nodes)
	void *v_arg;
	struct radix_node_head *head;
	int *dupentry;
	struct radix_node nodes[2];
{
	caddr_t v = v_arg;
	struct radix_node *top = head->rnh_treetop;
	int head_off = top->rn_off, vlen = (int)*((u_char *)v);
	register struct radix_node *t = rn_search(v_arg, top);
	register caddr_t cp = v + head_off;
	register int b;
	struct radix_node *tt;
    	/*
	 * Find first bit at which v and t->rn_key differ
	 */
    {
	register caddr_t cp2 = t->rn_key + head_off;
	register int cmp_res;
	caddr_t cplim = v + vlen;

	while (cp < cplim)
		if (*cp2++ != *cp++)
			goto on1;
	*dupentry = 1;
	return t;
on1:
	*dupentry = 0;
	cmp_res = (cp[-1] ^ cp2[-1]) & 0xff;
	for (b = (cp - v) << 3; cmp_res; b--)
		cmp_res >>= 1;
    }
    {
	register struct radix_node *p, *x = top;
	cp = v;
	do {
		p = x;
		if (cp[x->rn_off] & x->rn_bmask)
			x = x->rn_r;
		else x = x->rn_l;
	} while (b > (unsigned) x->rn_b); /* x->rn_b < b && x->rn_b >= 0 */
#ifdef RN_DEBUG
	if (rn_debug)
		log(LOG_DEBUG, "rn_insert: Going In:\n"), traverse(p);
#endif
	t = rn_newpair(v_arg, b, nodes); tt = t->rn_l;
	if ((cp[p->rn_off] & p->rn_bmask) == 0)
		p->rn_l = t;
	else
		p->rn_r = t;
	x->rn_p = t; t->rn_p = p; /* frees x, p as temp vars below */
	if ((cp[t->rn_off] & t->rn_bmask) == 0) {
		t->rn_r = x;
	} else {
		t->rn_r = tt; t->rn_l = x;
	}
#ifdef RN_DEBUG
	if (rn_debug)
		log(LOG_DEBUG, "rn_insert: Coming Out:\n"), traverse(p);
#endif
    }
	return (tt);
}

struct radix_node *
rn_addmask(n_arg, search, skip)
	int search, skip;
	void *n_arg;
{
	caddr_t netmask = (caddr_t)n_arg;
	register struct radix_node *x;
	register caddr_t cp, cplim;
	register int b = 0, mlen, j;
	int maskduplicated, m0, isnormal;
	struct radix_node *saved_x;
	static int last_zeroed = 0;

	if ((mlen = *(u_char *)netmask) > max_keylen)
		mlen = max_keylen;
	if (skip == 0)
		skip = 1;
	if (mlen <= skip)
		return (mask_rnhead->rnh_nodes);
	if (skip > 1)
		Bcopy(rn_ones + 1, addmask_key + 1, skip - 1);
	if ((m0 = mlen) > skip)
		Bcopy(netmask + skip, addmask_key + skip, mlen - skip);
	/*
	 * Trim trailing zeroes.
	 */
	for (cp = addmask_key + mlen; (cp > addmask_key) && cp[-1] == 0;)
		cp--;
	mlen = cp - addmask_key;
	if (mlen <= skip) {
		if (m0 >= last_zeroed)
			last_zeroed = mlen;
		return (mask_rnhead->rnh_nodes);
	}
	if (m0 < last_zeroed)
		Bzero(addmask_key + m0, last_zeroed - m0);
	*addmask_key = last_zeroed = mlen;
	x = rn_search(addmask_key, rn_masktop);
	if (Bcmp(addmask_key, x->rn_key, mlen) != 0)
		x = 0;
	if (x || search)
		return (x);
	R_Malloc(x, struct radix_node *, max_keylen + 2 * sizeof (*x));
	memUsed+= max_keylen + 2 * sizeof (*x);
	memForTree+= max_keylen + 2 * sizeof (*x);

	if ((saved_x = x) == 0)
		return (0);
	Bzero(x, max_keylen + 2 * sizeof (*x));
	netmask = cp = (caddr_t)(x + 2);
	Bcopy(addmask_key, cp, mlen);
	x = rn_insert(cp, mask_rnhead, &maskduplicated, x);
	if (maskduplicated) {
		log(LOG_ERR, "rn_addmask: mask impossibly already in tree");
		Free(saved_x);
		return (x);
	}
	/*
	 * Calculate index of mask, and check for normalcy.
	 */
	cplim = netmask + mlen; isnormal = 1;
	for (cp = netmask + skip; (cp < cplim) && *(u_char *)cp == 0xff;)
		cp++;
	if (cp != cplim) {
		for (j = 0x80; (j & *cp) != 0; j >>= 1)
			b++;
		if (*cp != normal_chars[b] || cp != (cplim - 1))
			isnormal = 0;
	}
	b += (cp - netmask) << 3;
	x->rn_b = -1 - b;
	if (isnormal)
		x->rn_flags |= RNF_NORMAL;
	return (x);
}

static int	/* XXX: arbitrary ordering for non-contiguous masks */
rn_lexobetter(m_arg, n_arg)
	void *m_arg, *n_arg;
{
	register u_char *mp = m_arg, *np = n_arg, *lim;

	if (*mp > *np)
		return 1;  /* not really, but need to check longer one first */
	if (*mp == *np)
		for (lim = mp + *mp; mp < lim;)
			if (*mp++ > *np++)
				return 1;
	return 0;
}

static struct radix_mask *
rn_new_radix_mask(tt, next)
	register struct radix_node *tt;
	register struct radix_mask *next;
{
	register struct radix_mask *m;

	MKGet(m);
	if (m == 0) {
		log(LOG_ERR, "Mask for route not entered\n");
		return (0);
	}
	Bzero(m, sizeof *m);
	m->rm_b = tt->rn_b;
	m->rm_flags = tt->rn_flags;
	if (tt->rn_flags & RNF_NORMAL)
		m->rm_leaf = tt;
	else
		m->rm_mask = tt->rn_mask;
	m->rm_mklist = next;
	tt->rn_mklist = m;
	return m;
}

struct radix_node *
rn_addroute(v_arg, n_arg, head, treenodes)
	void *v_arg, *n_arg;
	struct radix_node_head *head;
	struct radix_node treenodes[2];
{
	caddr_t v = (caddr_t)v_arg, netmask = (caddr_t)n_arg;
	register struct radix_node *t, *x = NULL, *tt;
	struct radix_node *saved_tt, *top = head->rnh_treetop;
	short b = 0, b_leaf = 0;
	int keyduplicated;
	caddr_t mmask;
	struct radix_mask *m, **mp;

	/*
	 * In dealing with non-contiguous masks, there may be
	 * many different routes which have the same mask.
	 * We will find it useful to have a unique pointer to
	 * the mask to speed avoiding duplicate references at
	 * nodes and possibly save time in calculating indices.
	 */
	if (netmask)  {
		if ((x = rn_addmask(netmask, 0, top->rn_off)) == 0)
			return (0);
		b_leaf = x->rn_b;
		b = -1 - x->rn_b;
		netmask = x->rn_key;
	}
	/*
	 * Deal with duplicated keys: attach node to previous instance
	 */
	saved_tt = tt = rn_insert(v, head, &keyduplicated, treenodes);
	if (keyduplicated) {
		for (t = tt; tt; t = tt, tt = tt->rn_dupedkey) {
			if (tt->rn_mask == netmask)
				return (0);
			if (netmask == 0 ||
			    (tt->rn_mask &&
			     ((b_leaf < tt->rn_b) || /* index(netmask) > node */
			       rn_refines(netmask, tt->rn_mask) ||
			       rn_lexobetter(netmask, tt->rn_mask))))
				break;
		}
		/*
		 * If the mask is not duplicated, we wouldn't
		 * find it among possible duplicate key entries
		 * anyway, so the above test doesn't hurt.
		 *
		 * We sort the masks for a duplicated key the same way as
		 * in a masklist -- most specific to least specific.
		 * This may require the unfortunate nuisance of relocating
		 * the head of the list.
		 */
		if (tt == saved_tt) {
			struct	radix_node *xx = x;
			/* link in at head of list */
			(tt = treenodes)->rn_dupedkey = t;
			tt->rn_flags = t->rn_flags;
			tt->rn_p = x = t->rn_p;
			if (x->rn_l == t) x->rn_l = tt; else x->rn_r = tt;
			saved_tt = tt; x = xx;
		} else {
			(tt = treenodes)->rn_dupedkey = t->rn_dupedkey;
			t->rn_dupedkey = tt;
		}
#ifdef RN_DEBUG
		t=tt+1; tt->rn_info = rn_nodenum++; t->rn_info = rn_nodenum++;
		tt->rn_twin = t; tt->rn_ybro = rn_clist; rn_clist = tt;
#endif
		tt->rn_key = (caddr_t) v;
		tt->rn_b = -1;
		tt->rn_flags = RNF_ACTIVE;
	}
	/*
	 * Put mask in tree.
	 */
	if (netmask) {
		tt->rn_mask = netmask;
		tt->rn_b = x->rn_b;
		tt->rn_flags |= x->rn_flags & RNF_NORMAL;
	}
	t = saved_tt->rn_p;
	if (keyduplicated)
		goto on2;
	b_leaf = -1 - t->rn_b;
	if (t->rn_r == saved_tt) x = t->rn_l; else x = t->rn_r;
	/* Promote general routes from below */
	if (x->rn_b < 0) {
	    for (mp = &t->rn_mklist; x; x = x->rn_dupedkey)
		if (x->rn_mask && (x->rn_b >= b_leaf) && x->rn_mklist == 0) {
			*mp = m = rn_new_radix_mask(x, 0);
			if (m)
				mp = &m->rm_mklist;
		}
	} else if (x->rn_mklist) {
		/*
		 * Skip over masks whose index is > that of new node
		 */
		for (mp = &x->rn_mklist; (m = *mp) != NULL; mp = &m->rm_mklist)
			if (m->rm_b >= b_leaf)
				break;
		t->rn_mklist = m; *mp = 0;
	}
on2:
	/* Add new route to highest possible ancestor's list */
	if ((netmask == 0) || (b > t->rn_b ))
		return tt; /* can't lift at all */
	b_leaf = tt->rn_b;
	do {
		x = t;
		t = t->rn_p;
	} while (b <= t->rn_b && x != top);
	/*
	 * Search through routes associated with node to
	 * insert new route according to index.
	 * Need same criteria as when sorting dupedkeys to avoid
	 * double loop on deletion.
	 */
	for (mp = &x->rn_mklist; (m = *mp) != NULL; mp = &m->rm_mklist) {
		if (m->rm_b < b_leaf)
			continue;
		if (m->rm_b > b_leaf)
			break;
		if (m->rm_flags & RNF_NORMAL) {
			mmask = m->rm_leaf->rn_mask;
			if (tt->rn_flags & RNF_NORMAL) {
				log(LOG_ERR,
				   "Non-unique normal route, mask not entered");
				return tt;
			}
		} else
			mmask = m->rm_mask;
		if (mmask == netmask) {
			m->rm_refs++;
			tt->rn_mklist = m;
			return tt;
		}
		if (rn_refines(netmask, mmask) || rn_lexobetter(netmask, mmask))
			break;
	}
	*mp = rn_new_radix_mask(tt, *mp);
	return tt;
}

struct radix_node *
rn_delete(v_arg, netmask_arg, head)
	void *v_arg, *netmask_arg;
	struct radix_node_head *head;
{
	register struct radix_node *t, *p, *x, *tt;
	struct radix_mask *m, *saved_m, **mp;
	struct radix_node *dupedkey, *saved_tt, *top;
	caddr_t v, netmask;
	int b, head_off, vlen;

	v = v_arg;
	netmask = netmask_arg;
	x = head->rnh_treetop;
	tt = rn_search(v, x);
	head_off = x->rn_off;
	vlen =  *(u_char *)v;
	saved_tt = tt;
	top = x;
	if (tt == 0 ||
	    Bcmp(v + head_off, tt->rn_key + head_off, vlen - head_off))
		return (0);
	/*
	 * Delete our route from mask lists.
	 */
	if (netmask) {
		if ((x = rn_addmask(netmask, 1, head_off)) == 0)
			return (0);
		netmask = x->rn_key;
		while (tt->rn_mask != netmask)
			if ((tt = tt->rn_dupedkey) == 0)
				return (0);
	}
	if (tt->rn_mask == 0 || (saved_m = m = tt->rn_mklist) == 0)
		goto on1;
	if (tt->rn_flags & RNF_NORMAL) {
		if (m->rm_leaf != tt || m->rm_refs > 0) {
			log(LOG_ERR, "rn_delete: inconsistent annotation\n");
			return 0;  /* dangling ref could cause disaster */
		}
	} else {
		if (m->rm_mask != tt->rn_mask) {
			log(LOG_ERR, "rn_delete: inconsistent annotation\n");
			goto on1;
		}
		if (--m->rm_refs >= 0)
			goto on1;
	}
	b = -1 - tt->rn_b;
	t = saved_tt->rn_p;
	if (b > t->rn_b)
		goto on1; /* Wasn't lifted at all */
	do {
		x = t;
		t = t->rn_p;
	} while (b <= t->rn_b && x != top);
	for (mp = &x->rn_mklist; (m = *mp) != NULL; mp = &m->rm_mklist)
		if (m == saved_m) {
			*mp = m->rm_mklist;
			MKFree(m);
			break;
		}
	if (m == 0) {
		log(LOG_ERR, "rn_delete: couldn't find our annotation\n");
		if (tt->rn_flags & RNF_NORMAL)
			return (0); /* Dangling ref to us */
	}
on1:
	/*
	 * Eliminate us from tree
	 */
	if (tt->rn_flags & RNF_ROOT)
		return (0);
#ifdef RN_DEBUG
	/* Get us out of the creation list */
	for (t = rn_clist; t && t->rn_ybro != tt; t = t->rn_ybro) {}
	if (t) t->rn_ybro = tt->rn_ybro;
#endif
	t = tt->rn_p;
	if ((dupedkey = saved_tt->rn_dupedkey) != 0) {
		if (tt == saved_tt) {
			x = dupedkey; x->rn_p = t;
			if (t->rn_l == tt) t->rn_l = x; else t->rn_r = x;
		} else {
			for (x = p = saved_tt; p && p->rn_dupedkey != tt;)
				p = p->rn_dupedkey;
			if (p) p->rn_dupedkey = tt->rn_dupedkey;
			else log(LOG_ERR, "rn_delete: couldn't find us\n");
		}
		t = tt + 1;
		if  (t->rn_flags & RNF_ACTIVE) {
#ifndef RN_DEBUG
			*++x = *t; p = t->rn_p;
#else
			b = t->rn_info; *++x = *t; t->rn_info = b; p = t->rn_p;
#endif
			if (p->rn_l == t) p->rn_l = x; else p->rn_r = x;
			x->rn_l->rn_p = x; x->rn_r->rn_p = x;
		}
		goto out;
	}
	if (t->rn_l == tt) x = t->rn_r; else x = t->rn_l;
	p = t->rn_p;
	if (p->rn_r == t) p->rn_r = x; else p->rn_l = x;
	x->rn_p = p;
	/*
	 * Demote routes attached to us.
	 */
	if (t->rn_mklist) {
		if (x->rn_b >= 0) {
			for (mp = &x->rn_mklist; (m = *mp) != NULL;)
				mp = &m->rm_mklist;
			*mp = t->rn_mklist;
		} else {
			/* If there are any key,mask pairs in a sibling
			   duped-key chain, some subset will appear sorted
			   in the same order attached to our mklist */
			for (m = t->rn_mklist; m && x; x = x->rn_dupedkey)
				if (m == x->rn_mklist) {
					struct radix_mask *mm = m->rm_mklist;
					x->rn_mklist = 0;
					if (--(m->rm_refs) < 0)
						MKFree(m);
					m = mm;
				}
			if (m)
				log5(LOG_ERR, "%s %p at %p\n",
					    "rn_delete: Orphaned Mask", m, x);
		}
	}
	/*
	 * We may be holding an active internal node in the tree.
	 */
	x = tt + 1;
	if (t != x) {
#ifndef RN_DEBUG
		*t = *x;
#else
		b = t->rn_info; *t = *x; t->rn_info = b;
#endif
		t->rn_l->rn_p = t; t->rn_r->rn_p = t;
		p = x->rn_p;
		if (p->rn_l == x) p->rn_l = t; else p->rn_r = t;
	}
out:
	tt->rn_flags &= ~RNF_ACTIVE;
	tt[1].rn_flags &= ~RNF_ACTIVE;
	return (tt);
}

int
rn_walktree(h, f, w)
	struct radix_node_head *h;
	register int (*f) (struct radix_node *, void *);
	void *w;
{
	int error;
	struct radix_node *base, *next;
	register struct radix_node *rn = h->rnh_treetop;
	/*
	 * This gets complicated because we may delete the node
	 * while applying the function f to it, so we need to calculate
	 * the successor node in advance.
	 */
	/* First time through node, go left */
	while (rn->rn_b >= 0)
		rn = rn->rn_l;
	for (;;) {
		base = rn;
		/* If at right child go back up, otherwise, go right */
		while (rn->rn_p->rn_r == rn && (rn->rn_flags & RNF_ROOT) == 0)
			rn = rn->rn_p;
		/* Find the next *leaf* since next node might vanish, too */
		for (rn = rn->rn_p->rn_r; rn->rn_b >= 0;)
			rn = rn->rn_l;
		next = rn;
		/* Process leaves */
		while ((rn = base) != NULL) {
			base = rn->rn_dupedkey;
			if (!(rn->rn_flags & RNF_ROOT) && (error = (*f)(rn, w)))
				return (error);
		}
		rn = next;
		if (rn->rn_flags & RNF_ROOT)
			return (0);
	}
	/* NOTREACHED */
}

int
rn_inithead(head, off)
	void **head;
	int off;
{
	register struct radix_node_head *rnh;
	register struct radix_node *t, *tt, *ttt;
	if (*head)
		return (1);
	R_Malloc(rnh, struct radix_node_head *, sizeof (*rnh));
	memUsed += sizeof (*rnh);
	memForTree += sizeof (*rnh);

	if (rnh == 0)
		return (0);
	Bzero(rnh, sizeof (*rnh));
	*head = rnh;
	t = rn_newpair(rn_zeros, off, rnh->rnh_nodes);
	ttt = rnh->rnh_nodes + 2;
	t->rn_r = ttt;
	t->rn_p = t;
	tt = t->rn_l;
	tt->rn_flags = t->rn_flags = RNF_ROOT | RNF_ACTIVE;
	tt->rn_b = -1 - off;
	*ttt = *tt;
	ttt->rn_key = rn_ones;
	rnh->rnh_addaddr = rn_addroute;
	rnh->rnh_deladdr = rn_delete;
	rnh->rnh_matchaddr = rn_match;
	rnh->rnh_lookup = rn_lookup;
	rnh->rnh_walktree = rn_walktree;
	rnh->rnh_treetop = t;
	return (1);
}

void
rn_init()
{
	char *cp, *cplim;
#ifdef _KERNEL
	struct domain *dom;

	for (dom = domains; dom; dom = dom->dom_next)
		if (dom->dom_maxrtkey > max_keylen)
			max_keylen = dom->dom_maxrtkey;
#endif
	if (max_keylen == 0) {
	  log(LOGERR,"rn_init: radix functions require max_keylen be set\n");
		return;
	}
	R_Malloc(rn_zeros, char *, 3 * max_keylen);
	memUsed += 3 * max_keylen;
	memForTree += 3 * max_keylen;
	if (rn_zeros == NULL)
	  {
		puts("panic in rn_init\n");
		exit(-1);
	  }
	Bzero(rn_zeros, 3 * max_keylen);
	rn_ones = cp = rn_zeros + max_keylen;
	addmask_key = cplim = rn_ones + max_keylen;
	while (cp < cplim)
		*cp++ = -1;
	if (rn_inithead((void **)&mask_rnhead, 0) == 0)
	  {
		puts("panic in rn_init 2\n");
		exit(-1);
	  }
}



void
rt_maskedcopy(src, dst, netmask)
	struct sockaddr *src, *dst, *netmask;
{
	register u_char *cp1 = (u_char *)src;
	register u_char *cp2 = (u_char *)dst;
	register u_char *cp3 = (u_char *)netmask;
	u_char *cplim = cp2 + *cp3;
	u_char *cplim2 = cp2 + *cp1;

	*cp2++ = *cp1++; *cp2++ = *cp1++; /* copies sa_len & sa_family */
	cp3 += 2;
	if (cplim > cplim2)
		cplim = cplim2;
	while (cp2 < cplim)
		*cp2++ = *cp1++ & *cp3++;
	if (cp2 < cplim2)
		bzero((caddr_t)cp2, (unsigned)(cplim2 - cp2));
}



#define N 32 /* Total number of bits in address*/
#define NBYTES N/8 /*No of bytes(assume N is a multiple of 8*/
#define FILENMLEN 100 /*Max len of filename*/

FILE *fp;
char fname[FILENMLEN];
char lname[FILENMLEN];

#define NOMASK 0
#define ISMASKED 1

/* len is length of the bit string in bytes */
/* Assuming the input file contains strings of the form a.b.c/x.y.z! and so on*/

int ReadAdr(FILE *fp,unsigned char *bytePtr,int *len,Info info,unsigned char *mask)
{
static int infoVal = 100; /*some general different value for each address*/
int retVal;

  int i,temp;
  int success;
  unsigned char c;

  retVal = ISMASKED;
  for ( i = 0 ; ; i++ )
    {
      success = fscanf(fp,"%d.",&temp);
      if (success == EOF) return EOF;
      if (!success) /*Means a / is encountered, read it up break*/
	{
	  fscanf(fp,"%c",&c); /* reads up the / */
	  break;
	}
      c = (unsigned char) temp;
      *bytePtr++ = c;
    }

  for (  ; (i<NBYTES) ; i++)
    *bytePtr++ = 0;

  /* ! is used to indicate end of the line*/
  if (mask!=NULL)
    {
      for (i = 0; ; i++)
	{
	  success = fscanf(fp,"%d.",&temp);
	  if (!success) /*if the ending ! is encountered*/
	    {
	      *mask = temp;
	      fscanf(fp,"%c",&c); /*read up the ! */
	      *len = i;
	      if ((i==NBYTES) && (temp==0xff)) retVal = NOMASK;
	      break;
	    }
	}
    }
  else
    {
      *len = NBYTES;
    }
  if (info!=NULL)/* when taking strings for searching from keyboard,
		    no info is given*/
    {
      info->i = infoVal++;
    }
  return retVal;
}


void PrintAdr(unsigned char *bytePtr,int startByte,int length,Info info)
{
  int i;

  for ( i = startByte; (i< startByte+length ) ; i++)
    {
      printf("%d.",bytePtr[i]);
    }

  if (info != NULL) printf(" (%d)",info->i);
  else puts("()");
}


struct sockaddr destination;

Info info;

unsigned char bitStr[NBYTES];

void gothrough(struct radix_node *rn,int level)
{
  int rnb;
  int i;
  struct radix_node *rn1;

  if  (rn!=NULL)
    {
      printf("%-8.8x ",rn);
      rnb = rn->rn_b;
      if (rnb >= 0)
	{
	  /*	  for ( i = 0; (i<level) ; i++) printf("  ");*/
	  printf("{");
	  printf("Level=%d bit=%d bmask=%d rn_off=%d",level,rnb,rn->rn_bmask,rn->rn_off);

	  printf("\n");
	  gothrough(rn->rn_l,level+1);
	  gothrough(rn->rn_r,level+1);
	  printf("}");
	}
      else
	{
	  /*	  for ( i = 0; (i<level) ; i++) printf("  ");*/
	  printf("{");
	  printf("Level=%d Leaf=%d bmask=%d key=",level,rnb,rn->rn_bmask);
	  if (rn->rn_flags & RNF_ROOT)
	    PrintAdr(rn->rn_key,0,4,NULL);
	  else 	PrintAdr(rn->rn_key,4,4,NULL);
	  for (rn1=rn; (rn1) ; rn1 = rn1->rn_dupedkey)
	    {
	      if ((rn1->rn_b < 0) && (rn1->rn_mask))
		{
		  printf("mask= ");
		  PrintAdr(rn1->rn_mask,4,4,NULL);
		}
	      else if ( rn1->rn_b >=0)
		{
		  printf("In duped key list got rn_b %d",rn1->rn_b);
		};
	      printf(" ");
	    }
	  printf("}");
	}
      printf("\n");
    }
}




static void
p_tree(rn)
	struct radix_node *rn;
{

if (rn == NULL) return;
again:
	if (rn->rn_b < 0)
	  {
	    printf("%-8.8x ", rn);
	    if (rn->rn_flags & RNF_ROOT)
	      {
		printf("(root node)%s",
		       rn->rn_dupedkey ? " =>\n" : "\n");
	      }
	    else
	      {
		PrintAdr(rn->rn_key,0,16,NULL);
		p_rtnode(rn);
	      }
	    if (rn = rn->rn_dupedkey)
	      goto again;
	  }
	else
	  {
	  printf("%-8.8x ", rn);
	  p_rtnode(rn);
	  p_tree(rn->rn_l);
	  p_tree(rn->rn_r);
	}
}


int Height(struct radix_node *rn)
{
int left,right;
if (rn == NULL) return 0;
if (rn->rn_b < 0)
  {
    return 1;
  }
else
  {
    if ((left=Height(rn->rn_l)) > (right=Height(rn->rn_r)))
      {
	return (1+left);
      }
    else return (1+right);
  }
}



char	nbuf[20];

static void
p_rtnode(struct radix_node *rn)
{
  struct radix_mask *rm = rn->rn_mklist;

  if (rn->rn_b < 0)
    {
    if (rn->rn_mask)
      {
      printf("\t  mask ");
      PrintAdr(rn->rn_mask, 0,16,NULL);
      }
    else if (rm == 0)
      return;
    }
  else
    {
    sprintf(nbuf, "(%d)", rn->rn_b);
    printf("%6.6s %8.8x : %8.8x", nbuf, rn->rn_l, rn->rn_r);
  }
  while (rm)
    {
    sprintf(nbuf, " %d refs, ", rm->rm_refs);
    printf(" mk = %8.8x {(%d),%s",
	   rm, -1 - rm->rm_b, rm->rm_refs ? nbuf : " ");
    PrintAdr(rm->rm_mask, 0, 16,NULL);
    putchar('}');
    if (rm = rm->rm_mklist)
      printf(" ->");
  }
  putchar('\n');
}


#define DIFSEC(tp2,tp1) ( (tp1).tv_sec - (tp2).tv_sec )
#define DIFUSEC(tp2,tp1)( (tp1).tv_usec -( tp2).tv_usec )
#define EFFUSEC(tp2,tp1) (DIFSEC(tp2,tp1)*1000000 + DIFUSEC(tp2, tp1))

int main(int argc, char *argv[]) {
  struct sockaddr *netmask,*netmask1,*ndst,*dst;
  struct sockaddr_in search;
  struct radix_node_head *rnh=NULL;
  register struct radix_node_head *rnh1=NULL;
  struct rtentry *rt;
  register struct radix_node *rn;

  register int *dstInt,*searchTable;

  struct timeval tp1,tp2;
  struct timezone tz;
  struct rusage ru1,ru2;
  int status,len,i,j,addr;
  unsigned char mask;
  int lookups, repeats;

  lookups=1000;
  repeats=1;

  while((i = getopt(argc,argv,"r:l:")) != EOF) {
    switch(i) {
    case 'r':  /* name of routing table entries */
      strcpy(fname, optarg);
      printf("routing file name %s.\n", fname);
      break;
    case 'l': /* number of different lookups */
      strcpy(lname, optarg);
      printf("lookup file name %s.\n", lname);
      break;
    default:
      printf("usage: %s -r routing_file -l lookup_file\n", argv[0]);
      exit(1);
    }
  }


  max_keylen = sizeof(struct sockaddr);
  rn_init();

  rn_inithead(&rnh,32);

  dst = &destination;
  dst->sa_len = sizeof(struct sockaddr);
  netmask = (struct sockaddr *)calloc(1,sizeof(struct sockaddr));
  memUsed+=sizeof(struct sockaddr);
  netmask->sa_len = sizeof(struct sockaddr);

  fp = fopen(fname,"r");
  if (fp == NULL)
    {
      printf("Cant open file %s\n",fname);
      exit(-1);
    }
  for ( ; (!(feof(fp))) ; )
    {
      info = (Info) calloc(1,sizeof(struct INFO));
      memUsed+=sizeof(struct INFO);
      status = ReadAdr(fp,bitStr,&len,info,&mask);
      if (status == EOF) break;


      R_Malloc(rt,struct rtentry *,sizeof(*rt));
      memUsed += sizeof(*rt);
      memForTree += sizeof(*rt);

      Bzero(rt,sizeof(*rt));
      rt_key(rt) = (struct sockaddr *)calloc(1,sizeof(struct sockaddr));
      memUsed += sizeof(struct sockaddr);
      memForTree += sizeof(struct sockaddr);

       ndst = rt_key(rt);
      rt->info = info;

      Bcopy(bitStr,((unsigned char *)(dst))+4,NBYTES);

      if (status == ISMASKED)
        {
          netmask = (struct sockaddr *)calloc(1,sizeof(struct sockaddr));
          memUsed += sizeof(struct sockaddr);

          netmask->sa_len = sizeof(struct sockaddr);
          rt_mask(rt) = (struct sockaddr *)calloc(1,sizeof(struct sockaddr));
          memUsed += sizeof(struct sockaddr);
          memForTree += sizeof(struct sockaddr);

          netmask1 = netmask;

          for (i=0; (i<len-1) ; i++) ((unsigned char *)netmask)[4+i]=0xff;
          ((unsigned char *)netmask)[4+len-1] = mask;
          for (i=len; i<NBYTES; i++) ((unsigned char *)netmask)[4+i]=0;

          rt_maskedcopy(dst, ndst, netmask);
        }
      else
        {

          netmask1 = NULL;
          Bcopy(dst, ndst, dst->sa_len);
        }
      rn = rn_addroute((caddr_t) ndst,(caddr_t)netmask1,rnh,rt->rt_nodes);
    }
  printf("Memory Used for radix tree= %d KB\n",memForTree/1000);
  printf("Memory Used totally = %d KB\n",memUsed/1000);
  printf("Height of radix tree = %d\n",Height(rnh->rnh_treetop));

  search.sin_len = 16;
  search.sin_family=AF_INET;
  search.sin_port = 0;
  bzero(search.sin_zero, 8);
  fclose(fp);

  fp = fopen(lname,"r");
  if (fp == NULL)
    {
      printf("Cant open file %s\n",fname);
      exit(-1);
    }
  for ( ; (!(feof(fp))) ; )
    {
      fscanf(fp, "%08x\n", &(search.sin_addr.s_addr));
      /*      printf("addr: %x\n", search.sin_addr.s_addr);
       */
      for (repeats=0; repeats<100; repeats++) {
	rn = rn_match(&search ,rnh);
      }
      /*
	 printf("addr = %x, rn = %x, rn_Key = %x\n", (int)search.sin_addr.s_addr,
	 (int)rn, (int)((struct sockaddr_in *)rn->rn_key)->sin_addr.s_addr);
      */
    }
  return 0;
}

